import { User } from '../index';

// User service specific types
export interface UserFilter {
  role?: User['role'];
  search?: string;
  isActive?: boolean;
  lastLoginRange?: {
    startDate: string;
    endDate: string;
  };
  department?: string;
}

export interface UserSortOptions {
  field: 'name' | 'email' | 'role' | 'createdAt' | 'lastLogin';
  order: 'asc' | 'desc';
}

export interface UserPaginationOptions {
  page: number;
  limit: number;
  sort?: UserSortOptions;
}

export interface UserListResponse {
  users: UserProfile[];
  total: number;
  page: number;
  limit: number;
  totalPages: number;
}

export interface UserProfile extends User {
  employeeId?: string;
  department?: string;
  position?: string;
  phone?: string;
  isActive: boolean;
  lastLogin?: string;
  createdAt: string;
  updatedAt: string;
  preferences: UserPreferences;
  permissions: UserPermission[];
}

export interface UserPreferences {
  theme: 'light' | 'dark' | 'system';
  language: string;
  timezone: string;
  dateFormat: 'MM/DD/YYYY' | 'DD/MM/YYYY' | 'YYYY-MM-DD';
  notifications: {
    email: boolean;
    browser: boolean;
    mobile: boolean;
    frequency: 'immediate' | 'daily' | 'weekly';
  };
  dashboard: {
    widgets: Array<{
      id: string;
      position: number;
      isVisible: boolean;
    }>;
    layout: 'grid' | 'list';
  };
}

export interface UserPermission {
  id: string;
  module: 'employees' | 'departments' | 'attendance' | 'leaves' | 'payroll' | 'reports' | 'admin';
  actions: ('create' | 'read' | 'update' | 'delete' | 'approve' | 'export')[];
  conditions?: {
    own_data_only?: boolean;
    department_only?: boolean;
    role_restrictions?: string[];
  };
}

export interface UserCreateRequest {
  email: string;
  name: string;
  role: User['role'];
  employeeId?: string;
  department?: string;
  position?: string;
  phone?: string;
  password: string;
  permissions?: string[]; // Permission IDs
  sendWelcomeEmail?: boolean;
}

export interface UserUpdateRequest {
  name?: string;
  role?: User['role'];
  department?: string;
  position?: string;
  phone?: string;
  avatar?: string;
  isActive?: boolean;
  permissions?: string[];
  preferences?: Partial<UserPreferences>;
}

export interface UserPasswordChangeRequest {
  currentPassword: string;
  newPassword: string;
  confirmPassword: string;
}

export interface UserPasswordResetRequest {
  email: string;
}

export interface UserPasswordResetConfirm {
  token: string;
  newPassword: string;
  confirmPassword: string;
}

export interface UserSession {
  id: string;
  userId: string;
  deviceInfo: {
    userAgent: string;
    ip: string;
    location?: string;
    deviceType: 'desktop' | 'mobile' | 'tablet';
  };
  loginAt: string;
  lastActivityAt: string;
  expiresAt: string;
  isActive: boolean;
}

export interface UserActivityLog {
  id: string;
  userId: string;
  action: string;
  module: string;
  details?: {
    resourceId?: string;
    resourceType?: string;
    changes?: any;
  };
  ipAddress: string;
  userAgent: string;
  timestamp: string;
  status: 'success' | 'failed';
}

export interface UserStats {
  total: number;
  active: number;
  inactive: number;
  byRole: {
    admin: number;
    hr: number;
    employee: number;
  };
  recentLogins: number; // Last 7 days
  sessionsActive: number;
  avgSessionDuration: number; // In minutes
}

// Enhanced Role Management Types
export interface Role {
  id: string;
  name: string;
  description: string;
  permissions: Permission[];
  isSystem: boolean; // System roles cannot be deleted
  isActive: boolean;
  createdAt: Date;
  updatedAt: Date;
  createdBy: string;
  userCount?: number;
}

export interface Permission {
  id: string;
  name: string;
  description: string;
  resource: string; // e.g., 'employees', 'payroll', 'reports'
  action: 'create' | 'read' | 'update' | 'delete' | 'execute';
  category: 'system' | 'hr' | 'employee' | 'finance' | 'reports';
}

export interface PermissionGroup {
  id: string;
  name: string;
  description: string;
  permissions: Permission[];
  category: string;
}

// Role CRUD Types
export interface RoleCreateRequest {
  name: string;
  description: string;
  permissionIds: string[];
}

export interface RoleUpdateRequest {
  name?: string;
  description?: string;
  permissionIds?: string[];
  isActive?: boolean;
}

export interface RoleFilter {
  search?: string;
  isActive?: boolean;
  isSystem?: boolean;
  hasPermission?: string;
}

export interface RoleListResponse {
  roles: Role[];
  total: number;
  page: number;
  limit: number;
  totalPages: number;
}

export interface RoleStats {
  total: number;
  system: number;
  custom: number;
  active: number;
  inactive: number;
}

export interface RolePaginationOptions {
  page: number;
  limit: number;
  sortBy?: 'name' | 'createdAt' | 'userCount';
  sortOrder?: 'asc' | 'desc';
}

// Legacy role interface for backward compatibility
export interface UserRole {
  id: string;
  name: string;
  description: string;
  permissions: UserPermission[];
  isDefault: boolean;
  isActive: boolean;
  userCount: number;
}

export interface UserInvitation {
  id: string;
  email: string;
  role: User['role'];
  invitedBy: string;
  invitedAt: string;
  expiresAt: string;
  status: 'pending' | 'accepted' | 'expired' | 'cancelled';
  acceptedAt?: string;
  token: string;
}

export interface UserBulkOperation {
  action: 'activate' | 'deactivate' | 'delete' | 'update_role' | 'reset_password';
  userIds: string[];
  params?: {
    newRole?: User['role'];
    sendNotification?: boolean;
  };
}

export interface UserBulkOperationResult {
  success: boolean;
  processedCount: number;
  failedCount: number;
  errors?: Array<{
    userId: string;
    error: string;
  }>;
}

export interface UserAuditLog {
  id: string;
  targetUserId: string;
  performedBy: string;
  action: 'created' | 'updated' | 'deleted' | 'activated' | 'deactivated' | 'password_reset' | 'role_changed';
  changes?: Array<{
    field: string;
    oldValue: any;
    newValue: any;
  }>;
  timestamp: string;
  ipAddress: string;
}

export interface UserLoginAttempt {
  id: string;
  email: string;
  ipAddress: string;
  userAgent: string;
  timestamp: string;
  success: boolean;
  failureReason?: 'invalid_credentials' | 'account_locked' | 'account_disabled' | 'too_many_attempts';
}

export interface UserSecuritySettings {
  userId: string;
  twoFactorEnabled: boolean;
  passwordLastChanged: string;
  failedLoginAttempts: number;
  accountLockedUntil?: string;
  trustedDevices: Array<{
    deviceId: string;
    deviceName: string;
    lastUsed: string;
    isActive: boolean;
  }>;
}

export interface UserNotificationSettings {
  userId: string;
  channels: {
    email: boolean;
    browser: boolean;
    mobile: boolean;
  };
  events: {
    leave_requests: boolean;
    attendance_alerts: boolean;
    payroll_updates: boolean;
    system_announcements: boolean;
    performance_reviews: boolean;
  };
  frequency: 'immediate' | 'daily' | 'weekly';
  quietHours: {
    enabled: boolean;
    start: string; // HH:MM
    end: string; // HH:MM
    timezone: string;
  };
}