import { User } from '../types';

// Mock data - replace with actual API calls
const mockUsers: User[] = [
  {
    id: '1',
    email: 'admin@hr.com',
    name: 'Admin User',
    role: 'admin',
    avatar: 'https://i.pravatar.cc/150?img=1'
  },
  {
    id: '2',
    email: 'hr@company.com',
    name: 'HR Manager',
    role: 'hr',
    avatar: 'https://i.pravatar.cc/150?img=2'
  },
  {
    id: '3',
    email: 'john.doe@company.com',
    name: 'John Doe',
    role: 'employee',
    avatar: 'https://i.pravatar.cc/150?img=3'
  },
  {
    id: '4',
    email: 'sarah.johnson@company.com',
    name: 'Sarah Johnson',
    role: 'employee',
    avatar: 'https://i.pravatar.cc/150?img=4'
  },
];

// Mock user credentials
const mockCredentials = [
  { email: 'admin@hr.com', password: 'admin123', userId: '1' },
  { email: 'hr@company.com', password: 'hr123', userId: '2' },
  { email: 'john.doe@company.com', password: 'john123', userId: '3' },
  { email: 'sarah.johnson@company.com', password: 'sarah123', userId: '4' },
];

let users = [...mockUsers];
let userSessions: { [key: string]: { userId: string; token: string; expiresAt: Date } } = {};

export const userService = {
  // Authenticate user
  authenticate: async (email: string, password: string): Promise<{
    success: boolean;
    user?: User;
    token?: string;
    message?: string;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const credential = mockCredentials.find(
          cred => cred.email === email && cred.password === password
        );
        
        if (credential) {
          const user = users.find(u => u.id === credential.userId);
          if (user) {
            const token = generateToken();
            const expiresAt = new Date();
            expiresAt.setHours(expiresAt.getHours() + 8); // 8-hour session
            
            userSessions[token] = {
              userId: user.id,
              token,
              expiresAt,
            };
            
            resolve({
              success: true,
              user,
              token,
            });
          } else {
            resolve({
              success: false,
              message: 'User not found',
            });
          }
        } else {
          resolve({
            success: false,
            message: 'Invalid email or password',
          });
        }
      }, 1000);
    });
  },

  // Validate session token
  validateToken: async (token: string): Promise<{
    valid: boolean;
    user?: User;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const session = userSessions[token];
        
        if (session && session.expiresAt > new Date()) {
          const user = users.find(u => u.id === session.userId);
          resolve({
            valid: true,
            user,
          });
        } else {
          if (session) {
            delete userSessions[token]; // Clean up expired session
          }
          resolve({
            valid: false,
          });
        }
      }, 300);
    });
  },

  // Logout user
  logout: async (token: string): Promise<{ success: boolean }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        if (userSessions[token]) {
          delete userSessions[token];
        }
        resolve({ success: true });
      }, 300);
    });
  },

  // Get all users
  getAllUsers: async (): Promise<User[]> => {
    return new Promise((resolve) => {
      setTimeout(() => resolve([...users]), 500);
    });
  },

  // Get user by ID
  getUserById: async (id: string): Promise<User | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const user = users.find(u => u.id === id);
        resolve(user || null);
      }, 300);
    });
  },

  // Get user by email
  getUserByEmail: async (email: string): Promise<User | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const user = users.find(u => u.email === email);
        resolve(user || null);
      }, 300);
    });
  },

  // Create new user
  createUser: async (userData: {
    email: string;
    name: string;
    role: 'admin' | 'hr' | 'employee';
    password: string;
    avatar?: string;
  }): Promise<{
    success: boolean;
    user?: User;
    message?: string;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        // Check if email already exists
        const existingUser = users.find(u => u.email === userData.email);
        if (existingUser) {
          resolve({
            success: false,
            message: 'Email already exists',
          });
          return;
        }

        const newUser: User = {
          id: Date.now().toString(),
          email: userData.email,
          name: userData.name,
          role: userData.role,
          avatar: userData.avatar,
        };

        users.push(newUser);
        
        // Add credentials
        mockCredentials.push({
          email: userData.email,
          password: userData.password,
          userId: newUser.id,
        });

        resolve({
          success: true,
          user: newUser,
        });
      }, 500);
    });
  },

  // Update user
  updateUser: async (id: string, userData: Partial<{
    name: string;
    role: 'admin' | 'hr' | 'employee';
    avatar: string;
  }>): Promise<{
    success: boolean;
    user?: User;
    message?: string;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = users.findIndex(u => u.id === id);
        if (index !== -1) {
          users[index] = { ...users[index], ...userData };
          resolve({
            success: true,
            user: users[index],
          });
        } else {
          resolve({
            success: false,
            message: 'User not found',
          });
        }
      }, 500);
    });
  },

  // Delete user
  deleteUser: async (id: string): Promise<{
    success: boolean;
    message?: string;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const userIndex = users.findIndex(u => u.id === id);
        if (userIndex !== -1) {
          const user = users[userIndex];
          
          // Remove user
          users.splice(userIndex, 1);
          
          // Remove credentials
          const credIndex = mockCredentials.findIndex(c => c.userId === id);
          if (credIndex !== -1) {
            mockCredentials.splice(credIndex, 1);
          }
          
          // Remove active sessions
          Object.keys(userSessions).forEach(token => {
            if (userSessions[token].userId === id) {
              delete userSessions[token];
            }
          });
          
          resolve({
            success: true,
            message: `User ${user.name} deleted successfully`,
          });
        } else {
          resolve({
            success: false,
            message: 'User not found',
          });
        }
      }, 500);
    });
  },

  // Change password
  changePassword: async (userId: string, currentPassword: string, newPassword: string): Promise<{
    success: boolean;
    message?: string;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const credential = mockCredentials.find(c => c.userId === userId);
        
        if (!credential) {
          resolve({
            success: false,
            message: 'User not found',
          });
          return;
        }
        
        if (credential.password !== currentPassword) {
          resolve({
            success: false,
            message: 'Current password is incorrect',
          });
          return;
        }
        
        credential.password = newPassword;
        
        resolve({
          success: true,
          message: 'Password changed successfully',
        });
      }, 500);
    });
  },

  // Reset password
  resetPassword: async (email: string): Promise<{
    success: boolean;
    message?: string;
    resetToken?: string;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const user = users.find(u => u.email === email);
        if (!user) {
          resolve({
            success: false,
            message: 'Email not found',
          });
          return;
        }
        
        // Generate reset token (in real app, this would be emailed)
        const resetToken = generateToken();
        
        resolve({
          success: true,
          message: 'Password reset instructions sent to your email',
          resetToken, // In real app, this wouldn't be returned directly
        });
      }, 1000);
    });
  },

  // Get users by role
  getUsersByRole: async (role: 'admin' | 'hr' | 'employee'): Promise<User[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = users.filter(u => u.role === role);
        resolve(filtered);
      }, 300);
    });
  },

  // Get user permissions
  getUserPermissions: async (userId: string): Promise<{
    canViewAllEmployees: boolean;
    canEditEmployees: boolean;
    canDeleteEmployees: boolean;
    canViewPayroll: boolean;
    canProcessPayroll: boolean;
    canApproveLeaves: boolean;
    canViewReports: boolean;
    canManageDepartments: boolean;
    canManageUsers: boolean;
    canViewDashboard: boolean;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const user = users.find(u => u.id === userId);
        
        if (!user) {
          resolve({
            canViewAllEmployees: false,
            canEditEmployees: false,
            canDeleteEmployees: false,
            canViewPayroll: false,
            canProcessPayroll: false,
            canApproveLeaves: false,
            canViewReports: false,
            canManageDepartments: false,
            canManageUsers: false,
            canViewDashboard: false,
          });
          return;
        }

        // Permission matrix based on role
        const permissions = {
          admin: {
            canViewAllEmployees: true,
            canEditEmployees: true,
            canDeleteEmployees: true,
            canViewPayroll: true,
            canProcessPayroll: true,
            canApproveLeaves: true,
            canViewReports: true,
            canManageDepartments: true,
            canManageUsers: true,
            canViewDashboard: true,
          },
          hr: {
            canViewAllEmployees: true,
            canEditEmployees: true,
            canDeleteEmployees: false,
            canViewPayroll: true,
            canProcessPayroll: false,
            canApproveLeaves: true,
            canViewReports: true,
            canManageDepartments: false,
            canManageUsers: false,
            canViewDashboard: true,
          },
          employee: {
            canViewAllEmployees: false,
            canEditEmployees: false,
            canDeleteEmployees: false,
            canViewPayroll: false,
            canProcessPayroll: false,
            canApproveLeaves: false,
            canViewReports: false,
            canManageDepartments: false,
            canManageUsers: false,
            canViewDashboard: false,
          },
        };

        resolve(permissions[user.role]);
      }, 300);
    });
  },

  // Get active sessions
  getActiveSessions: async (userId?: string): Promise<Array<{
    token: string;
    userId: string;
    userName: string;
    createdAt: Date;
    expiresAt: Date;
    isExpired: boolean;
  }>> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const sessions = Object.values(userSessions)
          .filter(session => !userId || session.userId === userId)
          .map(session => {
            const user = users.find(u => u.id === session.userId);
            return {
              token: session.token,
              userId: session.userId,
              userName: user?.name || 'Unknown',
              createdAt: new Date(session.expiresAt.getTime() - 8 * 60 * 60 * 1000), // 8 hours ago
              expiresAt: session.expiresAt,
              isExpired: session.expiresAt <= new Date(),
            };
          });

        resolve(sessions);
      }, 300);
    });
  },

  // Revoke session
  revokeSession: async (token: string): Promise<{ success: boolean }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        if (userSessions[token]) {
          delete userSessions[token];
        }
        resolve({ success: true });
      }, 300);
    });
  },

  // Get user activity log
  getUserActivityLog: async (userId: string, limit?: number): Promise<Array<{
    id: string;
    userId: string;
    action: string;
    resource: string;
    timestamp: string;
    ipAddress?: string;
    userAgent?: string;
  }>> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        // Mock activity log
        const activities = [
          {
            id: '1',
            userId,
            action: 'login',
            resource: 'authentication',
            timestamp: new Date().toISOString(),
            ipAddress: '192.168.1.100',
            userAgent: 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
          },
          {
            id: '2',
            userId,
            action: 'view',
            resource: 'employees',
            timestamp: new Date(Date.now() - 5 * 60 * 1000).toISOString(),
            ipAddress: '192.168.1.100',
          },
          {
            id: '3',
            userId,
            action: 'update',
            resource: 'employee:123',
            timestamp: new Date(Date.now() - 15 * 60 * 1000).toISOString(),
            ipAddress: '192.168.1.100',
          },
        ];

        const limitedActivities = limit ? activities.slice(0, limit) : activities;
        resolve(limitedActivities);
      }, 500);
    });
  },

  // Bulk user operations
  bulkUpdateUsers: async (updates: Array<{ id: string; data: Partial<User> }>): Promise<{
    success: boolean;
    updated: User[];
    failed: Array<{ id: string; error: string }>;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const updated: User[] = [];
        const failed: Array<{ id: string; error: string }> = [];

        updates.forEach(update => {
          const index = users.findIndex(u => u.id === update.id);
          if (index !== -1) {
            users[index] = { ...users[index], ...update.data };
            updated.push(users[index]);
          } else {
            failed.push({ id: update.id, error: 'User not found' });
          }
        });

        resolve({
          success: true,
          updated,
          failed,
        });
      }, 1000);
    });
  },
};

// Helper function to generate tokens
function generateToken(): string {
  return Date.now().toString(36) + Math.random().toString(36).substring(2);
}