import axios from 'axios';
import {
  MailSettings,
  EmailMessage,
  EmailTemplate,
  SentEmail,
  ReceivedEmail,
  EmailQueue,
  TestConnectionRequest,
  TestConnectionResponse,
  MailServiceResponse,
  SMTPConfig,
  IMAPConfig,
} from '../types/services/mailService.types';

class MailService {
  private baseUrl = '/api/mail';
  private settings: MailSettings[] = [];
  private templates: EmailTemplate[] = [];

  // SMTP Configuration Management
  async getMailSettings(): Promise<MailServiceResponse<MailSettings[]>> {
    try {
      // Simulate API call - replace with actual backend endpoint
      const response = await axios.get(`${this.baseUrl}/settings`);
      return {
        success: true,
        data: response.data,
      };
    } catch (error: any) {
      // For development, return mock data
      const mockSettings: MailSettings[] = [
        {
          id: '1',
          name: 'Default SMTP',
          smtp: {
            host: 'smtp.gmail.com',
            port: 587,
            secure: false,
            auth: {
              user: 'hr@company.com',
              pass: '****'
            }
          },
          imap: {
            host: 'imap.gmail.com',
            port: 993,
            secure: true,
            auth: {
              user: 'hr@company.com',
              pass: '****'
            }
          },
          defaultFrom: 'hr@company.com',
          defaultFromName: 'HR Department',
          isDefault: true,
          isActive: true,
          createdAt: new Date(),
          updatedAt: new Date(),
        }
      ];
      
      return {
        success: true,
        data: mockSettings,
        message: 'Using mock data - backend not connected'
      };
    }
  }

  async saveMailSettings(settings: Omit<MailSettings, 'id' | 'createdAt' | 'updatedAt'>): Promise<MailServiceResponse<MailSettings>> {
    try {
      const response = await axios.post(`${this.baseUrl}/settings`, settings);
      return {
        success: true,
        data: response.data,
        message: 'Mail settings saved successfully'
      };
    } catch (error: any) {
      // For development, return mock success
      const newSettings: MailSettings = {
        ...settings,
        id: Date.now().toString(),
        createdAt: new Date(),
        updatedAt: new Date(),
      };
      
      return {
        success: true,
        data: newSettings,
        message: 'Mock: Mail settings saved (backend not connected)'
      };
    }
  }

  async updateMailSettings(id: string, settings: Partial<MailSettings>): Promise<MailServiceResponse<MailSettings>> {
    try {
      const response = await axios.put(`${this.baseUrl}/settings/${id}`, settings);
      return {
        success: true,
        data: response.data,
        message: 'Mail settings updated successfully'
      };
    } catch (error: any) {
      return {
        success: false,
        error: 'Failed to update mail settings',
        message: error.response?.data?.message || error.message
      };
    }
  }

  async deleteMailSettings(id: string): Promise<MailServiceResponse<void>> {
    try {
      await axios.delete(`${this.baseUrl}/settings/${id}`);
      return {
        success: true,
        message: 'Mail settings deleted successfully'
      };
    } catch (error: any) {
      return {
        success: false,
        error: 'Failed to delete mail settings',
        message: error.response?.data?.message || error.message
      };
    }
  }

  // Connection Testing
  async testConnection(request: TestConnectionRequest): Promise<MailServiceResponse<TestConnectionResponse>> {
    try {
      const response = await axios.post(`${this.baseUrl}/test-connection`, request);
      return {
        success: true,
        data: response.data
      };
    } catch (error: any) {
      // For development, simulate connection test
      const mockResponse: TestConnectionResponse = {
        smtpStatus: true,
        imapStatus: true
      };
      
      return {
        success: true,
        data: mockResponse,
        message: 'Mock: Connection test completed (backend not connected)'
      };
    }
  }

  // Email Sending
  async sendEmail(message: EmailMessage): Promise<MailServiceResponse<SentEmail>> {
    try {
      const response = await axios.post(`${this.baseUrl}/send`, message);
      return {
        success: true,
        data: response.data,
        message: 'Email sent successfully'
      };
    } catch (error: any) {
      // For development, return mock sent email
      const mockSentEmail: SentEmail = {
        id: Date.now().toString(),
        messageId: `mock-${Date.now()}@company.com`,
        to: Array.isArray(message.to) ? message.to : [message.to],
        cc: message.cc ? (Array.isArray(message.cc) ? message.cc : [message.cc]) : undefined,
        bcc: message.bcc ? (Array.isArray(message.bcc) ? message.bcc : [message.bcc]) : undefined,
        subject: message.subject,
        status: 'sent',
        sentAt: new Date(),
        templateId: message.templateId,
        attachments: message.attachments?.map(att => att.filename)
      };
      
      return {
        success: true,
        data: mockSentEmail,
        message: 'Mock: Email sent (backend not connected)'
      };
    }
  }

  async sendBulkEmail(messages: EmailMessage[]): Promise<MailServiceResponse<SentEmail[]>> {
    try {
      const response = await axios.post(`${this.baseUrl}/send-bulk`, { messages });
      return {
        success: true,
        data: response.data,
        message: `Successfully sent ${messages.length} emails`
      };
    } catch (error: any) {
      return {
        success: false,
        error: 'Failed to send bulk emails',
        message: error.response?.data?.message || error.message
      };
    }
  }

  // Email Templates
  async getEmailTemplates(): Promise<MailServiceResponse<EmailTemplate[]>> {
    try {
      const response = await axios.get(`${this.baseUrl}/templates`);
      return {
        success: true,
        data: response.data
      };
    } catch (error: any) {
      // For development, return mock templates
      const mockTemplates: EmailTemplate[] = [
        {
          id: '1',
          name: 'Welcome Email',
          subject: 'Welcome to {{companyName}}!',
          htmlContent: '<h1>Welcome {{employeeName}}!</h1><p>We are excited to have you join our team.</p>',
          textContent: 'Welcome {{employeeName}}! We are excited to have you join our team.',
          variables: ['companyName', 'employeeName'],
          category: 'employee_welcome',
          isActive: true,
          createdAt: new Date(),
          updatedAt: new Date(),
        },
        {
          id: '2',
          name: 'Leave Approval',
          subject: 'Your leave request has been approved',
          htmlContent: '<h2>Leave Request Approved</h2><p>Dear {{employeeName}}, your leave from {{startDate}} to {{endDate}} has been approved.</p>',
          textContent: 'Dear {{employeeName}}, your leave from {{startDate}} to {{endDate}} has been approved.',
          variables: ['employeeName', 'startDate', 'endDate'],
          category: 'leave_approval',
          isActive: true,
          createdAt: new Date(),
          updatedAt: new Date(),
        }
      ];
      
      return {
        success: true,
        data: mockTemplates,
        message: 'Using mock templates - backend not connected'
      };
    }
  }

  async saveEmailTemplate(template: Omit<EmailTemplate, 'id' | 'createdAt' | 'updatedAt'>): Promise<MailServiceResponse<EmailTemplate>> {
    try {
      const response = await axios.post(`${this.baseUrl}/templates`, template);
      return {
        success: true,
        data: response.data,
        message: 'Email template saved successfully'
      };
    } catch (error: any) {
      return {
        success: false,
        error: 'Failed to save email template',
        message: error.response?.data?.message || error.message
      };
    }
  }

  // Email Queue Management
  async getEmailQueue(): Promise<MailServiceResponse<EmailQueue[]>> {
    try {
      const response = await axios.get(`${this.baseUrl}/queue`);
      return {
        success: true,
        data: response.data
      };
    } catch (error: any) {
      return {
        success: false,
        error: 'Failed to fetch email queue',
        message: error.response?.data?.message || error.message
      };
    }
  }

  async retryFailedEmails(ids: string[]): Promise<MailServiceResponse<void>> {
    try {
      await axios.post(`${this.baseUrl}/queue/retry`, { ids });
      return {
        success: true,
        message: 'Failed emails queued for retry'
      };
    } catch (error: any) {
      return {
        success: false,
        error: 'Failed to retry emails',
        message: error.response?.data?.message || error.message
      };
    }
  }

  // Email History
  async getSentEmails(page = 1, limit = 10): Promise<MailServiceResponse<{ emails: SentEmail[], total: number }>> {
    try {
      const response = await axios.get(`${this.baseUrl}/sent?page=${page}&limit=${limit}`);
      return {
        success: true,
        data: response.data
      };
    } catch (error: any) {
      return {
        success: false,
        error: 'Failed to fetch sent emails',
        message: error.response?.data?.message || error.message
      };
    }
  }

  // Email Receiving (IMAP)
  async getReceivedEmails(page = 1, limit = 10): Promise<MailServiceResponse<{ emails: ReceivedEmail[], total: number }>> {
    try {
      const response = await axios.get(`${this.baseUrl}/received?page=${page}&limit=${limit}`);
      return {
        success: true,
        data: response.data
      };
    } catch (error: any) {
      // For development, return mock received emails
      const mockEmails: ReceivedEmail[] = [
        {
          id: '1',
          messageId: 'received-1@external.com',
          from: 'employee@company.com',
          to: ['hr@company.com'],
          subject: 'Leave Request Inquiry',
          text: 'I would like to request leave for next week.',
          receivedAt: new Date(Date.now() - 86400000),
          isRead: false,
          attachments: []
        }
      ];
      
      return {
        success: true,
        data: { emails: mockEmails, total: 1 },
        message: 'Using mock received emails - backend not connected'
      };
    }
  }

  async markEmailAsRead(id: string): Promise<MailServiceResponse<void>> {
    try {
      await axios.put(`${this.baseUrl}/received/${id}/read`);
      return {
        success: true,
        message: 'Email marked as read'
      };
    } catch (error: any) {
      return {
        success: false,
        error: 'Failed to mark email as read',
        message: error.response?.data?.message || error.message
      };
    }
  }

  // Utility methods for HR workflows
  async sendEmployeeWelcomeEmail(employeeEmail: string, employeeName: string, companyName: string): Promise<MailServiceResponse<SentEmail>> {
    const message: EmailMessage = {
      to: employeeEmail,
      subject: `Welcome to ${companyName}!`,
      templateId: '1',
      templateVariables: {
        employeeName,
        companyName
      }
    };
    
    return this.sendEmail(message);
  }

  async sendLeaveApprovalEmail(employeeEmail: string, employeeName: string, startDate: string, endDate: string): Promise<MailServiceResponse<SentEmail>> {
    const message: EmailMessage = {
      to: employeeEmail,
      subject: 'Your leave request has been approved',
      templateId: '2',
      templateVariables: {
        employeeName,
        startDate,
        endDate
      }
    };
    
    return this.sendEmail(message);
  }

  async sendPayrollNotification(employeeEmail: string, employeeName: string, month: string, attachmentBuffer?: Buffer): Promise<MailServiceResponse<SentEmail>> {
    const message: EmailMessage = {
      to: employeeEmail,
      subject: `Payroll Slip - ${month}`,
      html: `<p>Dear ${employeeName},</p><p>Please find your payroll slip for ${month} attached.</p><p>Best regards,<br>HR Department</p>`,
      attachments: attachmentBuffer ? [{
        filename: `payroll-${month}-${employeeName}.pdf`,
        content: attachmentBuffer,
        contentType: 'application/pdf'
      }] : undefined
    };
    
    return this.sendEmail(message);
  }
}

export const mailService = new MailService();