import {
  Holiday,
  HolidayFilter,
  HolidayListResponse,
  HolidayCreateRequest,
  HolidayUpdateRequest,
  HolidayStats,
  HolidayPaginationOptions,
  HolidayCalendarView,
  HolidayBulkOperation,
  HolidayBulkOperationResult,
  HolidayImportOptions,
  HolidayExportOptions,
  HolidayConflict,
  IHolidayService,
  ServiceResponse,
} from '../types';

// Mock data for development
const mockHolidays: Holiday[] = [
  {
    id: '1',
    name: 'New Year\'s Day',
    date: '2024-01-01',
    description: 'First day of the year',
    type: 'public',
    isRecurring: true,
    organizationId: 'org-1',
    color: '#ff4444',
    createdAt: new Date('2023-12-01'),
    updatedAt: new Date('2023-12-01'),
    createdBy: 'admin-1'
  },
  {
    id: '2',
    name: 'Independence Day',
    date: '2024-07-04',
    description: 'National holiday celebrating independence',
    type: 'public',
    isRecurring: true,
    organizationId: 'org-1',
    color: '#4444ff',
    createdAt: new Date('2023-12-01'),
    updatedAt: new Date('2023-12-01'),
    createdBy: 'admin-1'
  },
  {
    id: '3',
    name: 'Company Foundation Day',
    date: '2024-03-15',
    description: 'Celebrating our company foundation',
    type: 'company',
    isRecurring: true,
    organizationId: 'org-1',
    departmentIds: ['dept-1', 'dept-2'],
    color: '#44ff44',
    createdAt: new Date('2023-12-01'),
    updatedAt: new Date('2023-12-01'),
    createdBy: 'hr-1'
  },
  {
    id: '4',
    name: 'Christmas',
    date: '2024-12-25',
    description: 'Christmas celebration',
    type: 'religious',
    isRecurring: true,
    organizationId: 'org-1',
    color: '#ff44ff',
    createdAt: new Date('2023-12-01'),
    updatedAt: new Date('2023-12-01'),
    createdBy: 'admin-1'
  },
  {
    id: '5',
    name: 'Good Friday',
    date: '2024-03-29',
    description: 'Good Friday observance',
    type: 'optional',
    isRecurring: true,
    organizationId: 'org-1',
    color: '#ffff44',
    createdAt: new Date('2023-12-01'),
    updatedAt: new Date('2023-12-01'),
    createdBy: 'admin-1'
  }
];

let holidays = [...mockHolidays];

class HolidayService implements IHolidayService {
  private delay = (ms: number) => new Promise(resolve => setTimeout(resolve, ms));

  async getAllHolidays(
    filter?: HolidayFilter,
    pagination?: HolidayPaginationOptions
  ): Promise<HolidayListResponse> {
    await this.delay(500);
    
    let filteredHolidays = [...holidays];
    
    // Apply filters
    if (filter) {
      if (filter.year) {
        filteredHolidays = filteredHolidays.filter(holiday =>
          new Date(holiday.date).getFullYear() === filter.year
        );
      }
      
      if (filter.month) {
        filteredHolidays = filteredHolidays.filter(holiday =>
          new Date(holiday.date).getMonth() + 1 === filter.month
        );
      }
      
      if (filter.type) {
        filteredHolidays = filteredHolidays.filter(holiday =>
          holiday.type === filter.type
        );
      }
      
      if (filter.organizationId) {
        filteredHolidays = filteredHolidays.filter(holiday =>
          holiday.organizationId === filter.organizationId
        );
      }
      
      if (filter.search) {
        const searchTerm = filter.search.toLowerCase();
        filteredHolidays = filteredHolidays.filter(holiday =>
          holiday.name.toLowerCase().includes(searchTerm) ||
          (holiday.description && holiday.description.toLowerCase().includes(searchTerm))
        );
      }
    }
    
    // Apply pagination
    const page = pagination?.page || 1;
    const limit = pagination?.limit || 10;
    const startIndex = (page - 1) * limit;
    const endIndex = startIndex + limit;
    
    // Apply sorting
    if (pagination?.sortBy) {
      const sortBy = pagination.sortBy;
      const sortOrder = pagination.sortOrder || 'asc';
      
      filteredHolidays.sort((a, b) => {
        let aValue: any = (a as any)[sortBy];
        let bValue: any = (b as any)[sortBy];
        
        if (sortBy === 'date') {
          aValue = new Date(aValue);
          bValue = new Date(bValue);
        }
        
        if (aValue < bValue) return sortOrder === 'asc' ? -1 : 1;
        if (aValue > bValue) return sortOrder === 'asc' ? 1 : -1;
        return 0;
      });
    }
    
    const paginatedHolidays = filteredHolidays.slice(startIndex, endIndex);
    
    return {
      holidays: paginatedHolidays,
      total: filteredHolidays.length,
      page,
      limit
    };
  }

  async getHolidayById(id: string): Promise<Holiday | null> {
    await this.delay(300);
    return holidays.find(holiday => holiday.id === id) || null;
  }

  async getHolidaysByDateRange(
    startDate: string,
    endDate: string,
    organizationId?: string
  ): Promise<Holiday[]> {
    await this.delay(400);
    
    const start = new Date(startDate);
    const end = new Date(endDate);
    
    return holidays.filter(holiday => {
      const holidayDate = new Date(holiday.date);
      const inRange = holidayDate >= start && holidayDate <= end;
      const orgMatch = !organizationId || holiday.organizationId === organizationId;
      return inRange && orgMatch;
    });
  }

  async createHoliday(holiday: HolidayCreateRequest): Promise<Holiday> {
    await this.delay(500);
    
    const newHoliday: Holiday = {
      id: `holiday-${Date.now()}`,
      ...holiday,
      createdAt: new Date(),
      updatedAt: new Date(),
      createdBy: 'current-user', // In real app, get from context
    };
    
    holidays.push(newHoliday);
    return newHoliday;
  }

  async updateHoliday(id: string, holidayUpdate: HolidayUpdateRequest): Promise<Holiday | null> {
    await this.delay(500);
    
    const index = holidays.findIndex(holiday => holiday.id === id);
    if (index === -1) return null;
    
    holidays[index] = {
      ...holidays[index],
      ...holidayUpdate,
      updatedAt: new Date(),
    };
    
    return holidays[index];
  }

  async deleteHoliday(id: string): Promise<boolean> {
    await this.delay(400);
    
    const index = holidays.findIndex(holiday => holiday.id === id);
    if (index === -1) return false;
    
    holidays.splice(index, 1);
    return true;
  }

  async getHolidayCalendarView(
    year: number,
    month: number,
    organizationId?: string
  ): Promise<HolidayCalendarView> {
    await this.delay(400);
    
    const startDate = new Date(year, month - 1, 1);
    const endDate = new Date(year, month, 0);
    
    const monthHolidays = holidays.filter(holiday => {
      const holidayDate = new Date(holiday.date);
      const inMonth = holidayDate >= startDate && holidayDate <= endDate;
      const orgMatch = !organizationId || holiday.organizationId === organizationId;
      return inMonth && orgMatch;
    });
    
    // Generate weekend dates for the month
    const weekends: string[] = [];
    for (let day = 1; day <= endDate.getDate(); day++) {
      const date = new Date(year, month - 1, day);
      if (date.getDay() === 0 || date.getDay() === 6) { // Sunday or Saturday
        weekends.push(date.toISOString().split('T')[0]);
      }
    }
    
    return {
      year,
      month,
      holidays: monthHolidays,
      weekends
    };
  }

  async getHolidaysForYear(year: number, organizationId?: string): Promise<Holiday[]> {
    await this.delay(400);
    
    return holidays.filter(holiday => {
      const holidayYear = new Date(holiday.date).getFullYear();
      const orgMatch = !organizationId || holiday.organizationId === organizationId;
      return holidayYear === year && orgMatch;
    });
  }

  async getUpcomingHolidays(limit = 5, organizationId?: string): Promise<Holiday[]> {
    await this.delay(300);
    
    const today = new Date();
    const upcomingHolidays = holidays
      .filter(holiday => {
        const holidayDate = new Date(holiday.date);
        const orgMatch = !organizationId || holiday.organizationId === organizationId;
        return holidayDate >= today && orgMatch;
      })
      .sort((a, b) => new Date(a.date).getTime() - new Date(b.date).getTime())
      .slice(0, limit);
    
    return upcomingHolidays;
  }

  async bulkCreateHolidays(holidayRequests: HolidayCreateRequest[]): Promise<HolidayBulkOperationResult> {
    await this.delay(800);
    
    let created = 0;
    const errors: string[] = [];
    
    for (const holidayRequest of holidayRequests) {
      try {
        await this.createHoliday(holidayRequest);
        created++;
      } catch (error) {
        errors.push(`Failed to create holiday ${holidayRequest.name}: ${error}`);
      }
    }
    
    return {
      success: errors.length === 0,
      created,
      failed: errors.length,
      errors
    };
  }

  async bulkUpdateHolidays(operation: HolidayBulkOperation): Promise<HolidayBulkOperationResult> {
    await this.delay(800);
    
    let updated = 0;
    const errors: string[] = [];
    
    if (operation.operation === 'update' && Array.isArray(operation.holidays)) {
      for (const holiday of operation.holidays as Holiday[]) {
        try {
          await this.updateHoliday(holiday.id, holiday);
          updated++;
        } catch (error) {
          errors.push(`Failed to update holiday ${holiday.name}: ${error}`);
        }
      }
    }
    
    return {
      success: errors.length === 0,
      updated,
      failed: errors.length,
      errors
    };
  }

  async bulkDeleteHolidays(holidayIds: string[]): Promise<HolidayBulkOperationResult> {
    await this.delay(600);
    
    let deleted = 0;
    const errors: string[] = [];
    
    for (const id of holidayIds) {
      try {
        const success = await this.deleteHoliday(id);
        if (success) {
          deleted++;
        } else {
          errors.push(`Holiday with ID ${id} not found`);
        }
      } catch (error) {
        errors.push(`Failed to delete holiday ${id}: ${error}`);
      }
    }
    
    return {
      success: errors.length === 0,
      deleted,
      failed: errors.length,
      errors
    };
  }

  async importHolidays(options: HolidayImportOptions): Promise<HolidayBulkOperationResult> {
    await this.delay(1000);
    
    // Mock implementation
    return {
      success: true,
      created: 5,
      failed: 0,
      errors: []
    };
  }

  async exportHolidays(options: HolidayExportOptions): Promise<any> {
    await this.delay(800);
    
    let exportHolidays = [...holidays];
    
    if (options.filter) {
      // Apply filters similar to getAllHolidays
      if (options.filter.year) {
        exportHolidays = exportHolidays.filter(holiday =>
          new Date(holiday.date).getFullYear() === options.filter!.year
        );
      }
      
      if (options.filter.type) {
        exportHolidays = exportHolidays.filter(holiday =>
          holiday.type === options.filter!.type
        );
      }
    }
    
    if (options.format === 'csv') {
      const csvHeaders = ['Name', 'Date', 'Type', 'Description'];
      const csvRows = exportHolidays.map(holiday => [
        holiday.name,
        holiday.date,
        holiday.type,
        holiday.description || ''
      ]);
      
      return {
        format: 'csv',
        data: [csvHeaders, ...csvRows],
        filename: `holidays_${new Date().toISOString().split('T')[0]}.csv`
      };
    }
    
    return {
      format: 'json',
      data: exportHolidays,
      filename: `holidays_${new Date().toISOString().split('T')[0]}.json`
    };
  }

  async validateHoliday(holiday: HolidayCreateRequest): Promise<{ isValid: boolean; errors: string[] }> {
    await this.delay(200);
    
    const errors: string[] = [];
    
    if (!holiday.name || holiday.name.trim().length === 0) {
      errors.push('Holiday name is required');
    }
    
    if (!holiday.date) {
      errors.push('Holiday date is required');
    } else {
      const date = new Date(holiday.date);
      if (isNaN(date.getTime())) {
        errors.push('Invalid date format');
      }
    }
    
    if (!holiday.type) {
      errors.push('Holiday type is required');
    }
    
    return {
      isValid: errors.length === 0,
      errors
    };
  }

  async checkHolidayConflicts(holiday: HolidayCreateRequest): Promise<HolidayConflict[]> {
    await this.delay(300);
    
    const conflicts: HolidayConflict[] = [];
    const holidayDate = new Date(holiday.date);
    
    const existingHolidays = holidays.filter(existingHoliday => {
      const existingDate = new Date(existingHoliday.date);
      return existingDate.toDateString() === holidayDate.toDateString();
    });
    
    if (existingHolidays.length > 0) {
      conflicts.push({
        date: holiday.date,
        holidays: existingHolidays,
        type: 'duplicate'
      });
    }
    
    return conflicts;
  }

  async getHolidayStats(filter?: HolidayFilter): Promise<HolidayStats> {
    await this.delay(400);
    
    let statsHolidays = [...holidays];
    
    if (filter) {
      if (filter.year) {
        statsHolidays = statsHolidays.filter(holiday =>
          new Date(holiday.date).getFullYear() === filter.year
        );
      }
      
      if (filter.organizationId) {
        statsHolidays = statsHolidays.filter(holiday =>
          holiday.organizationId === filter.organizationId
        );
      }
    }
    
    const byType = statsHolidays.reduce((acc, holiday) => {
      acc[holiday.type] = (acc[holiday.type] || 0) + 1;
      return acc;
    }, {} as Record<string, number>);
    
    const today = new Date();
    const endOfMonth = new Date(today.getFullYear(), today.getMonth() + 1, 0);
    
    const upcoming = statsHolidays.filter(holiday =>
      new Date(holiday.date) > today
    ).length;
    
    const thisMonth = statsHolidays.filter(holiday => {
      const holidayDate = new Date(holiday.date);
      return holidayDate.getMonth() === today.getMonth() &&
             holidayDate.getFullYear() === today.getFullYear();
    }).length;
    
    const recurring = statsHolidays.filter(holiday => holiday.isRecurring).length;
    
    return {
      total: statsHolidays.length,
      byType,
      upcoming,
      thisMonth,
      recurring
    };
  }

  async getHolidaysByType(type: string, year?: number): Promise<Holiday[]> {
    await this.delay(300);
    
    return holidays.filter(holiday => {
      const typeMatch = holiday.type === type;
      const yearMatch = !year || new Date(holiday.date).getFullYear() === year;
      return typeMatch && yearMatch;
    });
  }

  async getOrganizationHolidays(organizationId: string, year?: number): Promise<Holiday[]> {
    await this.delay(300);
    
    return holidays.filter(holiday => {
      const orgMatch = holiday.organizationId === organizationId;
      const yearMatch = !year || new Date(holiday.date).getFullYear() === year;
      return orgMatch && yearMatch;
    });
  }

  async getDepartmentHolidays(departmentIds: string[], year?: number): Promise<Holiday[]> {
    await this.delay(300);
    
    return holidays.filter(holiday => {
      const deptMatch = holiday.departmentIds?.some(id => departmentIds.includes(id));
      const yearMatch = !year || new Date(holiday.date).getFullYear() === year;
      return deptMatch && yearMatch;
    });
  }
}

export const holidayService = new HolidayService();