import { Employee } from '../types';
import { mailService } from './mailService';

// Mock data - replace with actual API calls
const mockEmployees: Employee[] = [
  {
    id: '1',
    employeeId: 'EMP001',
    firstName: 'John',
    lastName: 'Doe',
    email: 'john.doe@company.com',
    phone: '+1-555-0123',
    department: 'Engineering',
    position: 'Senior Software Engineer',
    salary: 95000,
    hireDate: '2023-01-15',
    status: 'active',
    address: '123 Main St, City, State 12345',
    dateOfBirth: '1990-05-15',
    emergencyContact: {
      name: 'Jane Doe',
      phone: '+1-555-0124',
      relationship: 'Spouse'
    }
  },
  {
    id: '2',
    employeeId: 'EMP002',
    firstName: 'Sarah',
    lastName: 'Johnson',
    email: 'sarah.johnson@company.com',
    phone: '+1-555-0125',
    department: 'Sales',
    position: 'Sales Manager',
    salary: 75000,
    hireDate: '2022-08-20',
    status: 'active',
    address: '456 Oak Ave, City, State 12345',
    dateOfBirth: '1988-12-03',
    emergencyContact: {
      name: 'Mike Johnson',
      phone: '+1-555-0126',
      relationship: 'Brother'
    }
  },
  {
    id: '3',
    employeeId: 'EMP003',
    firstName: 'Michael',
    lastName: 'Brown',
    email: 'michael.brown@company.com',
    phone: '+1-555-0127',
    department: 'Marketing',
    position: 'Marketing Specialist',
    salary: 62000,
    hireDate: '2023-03-10',
    status: 'active',
    address: '789 Pine St, City, State 12345',
    dateOfBirth: '1992-09-18',
    emergencyContact: {
      name: 'Lisa Brown',
      phone: '+1-555-0128',
      relationship: 'Mother'
    }
  },
  {
    id: '4',
    employeeId: 'EMP004',
    firstName: 'Emily',
    lastName: 'Davis',
    email: 'emily.davis@company.com',
    phone: '+1-555-0129',
    department: 'HR',
    position: 'HR Coordinator',
    salary: 58000,
    hireDate: '2022-11-05',
    status: 'active',
    address: '321 Elm St, City, State 12345',
    dateOfBirth: '1991-07-22',
    emergencyContact: {
      name: 'Robert Davis',
      phone: '+1-555-0130',
      relationship: 'Father'
    }
  },
  {
    id: '5',
    employeeId: 'EMP005',
    firstName: 'David',
    lastName: 'Wilson',
    email: 'david.wilson@company.com',
    phone: '+1-555-0131',
    department: 'Finance',
    position: 'Financial Analyst',
    salary: 68000,
    hireDate: '2023-02-28',
    status: 'active',
    address: '654 Maple Dr, City, State 12345',
    dateOfBirth: '1989-11-12',
    emergencyContact: {
      name: 'Amy Wilson',
      phone: '+1-555-0132',
      relationship: 'Wife'
    }
  }
];

let employees = [...mockEmployees];

export const employeeService = {
  // Get all employees
  getAllEmployees: async (): Promise<Employee[]> => {
    return new Promise((resolve) => {
      setTimeout(() => resolve([...employees]), 500);
    });
  },

  // Get employee by ID
  getEmployeeById: async (id: string): Promise<Employee | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const employee = employees.find(emp => emp.id === id);
        resolve(employee || null);
      }, 300);
    });
  },

  // Create new employee
  createEmployee: async (employeeData: Omit<Employee, 'id'>): Promise<Employee> => {
    return new Promise(async (resolve) => {
      setTimeout(async () => {
        const newEmployee: Employee = {
          ...employeeData,
          id: Date.now().toString(),
        };
        employees.push(newEmployee);
        
        // Send welcome email
        try {
          await mailService.sendEmployeeWelcomeEmail(
            newEmployee.email,
            `${newEmployee.firstName} ${newEmployee.lastName}`,
            'Your Company Name'
          );
        } catch (error) {
          console.error('Failed to send welcome email:', error);
        }
        
        resolve(newEmployee);
      }, 500);
    });
  },

  // Update employee
  updateEmployee: async (id: string, employeeData: Partial<Employee>): Promise<Employee | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = employees.findIndex(emp => emp.id === id);
        if (index !== -1) {
          employees[index] = { ...employees[index], ...employeeData };
          resolve(employees[index]);
        } else {
          resolve(null);
        }
      }, 500);
    });
  },

  // Delete employee
  deleteEmployee: async (id: string): Promise<boolean> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = employees.findIndex(emp => emp.id === id);
        if (index !== -1) {
          employees.splice(index, 1);
          resolve(true);
        } else {
          resolve(false);
        }
      }, 300);
    });
  },

  // Search employees
  searchEmployees: async (query: string): Promise<Employee[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = employees.filter(emp => 
          emp.firstName.toLowerCase().includes(query.toLowerCase()) ||
          emp.lastName.toLowerCase().includes(query.toLowerCase()) ||
          emp.email.toLowerCase().includes(query.toLowerCase()) ||
          emp.employeeId.toLowerCase().includes(query.toLowerCase()) ||
          emp.department.toLowerCase().includes(query.toLowerCase()) ||
          emp.position.toLowerCase().includes(query.toLowerCase())
        );
        resolve(filtered);
      }, 300);
    });
  },

  // Get employees by department
  getEmployeesByDepartment: async (department: string): Promise<Employee[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = employees.filter(emp => 
          emp.department.toLowerCase() === department.toLowerCase()
        );
        resolve(filtered);
      }, 300);
    });
  }
};