import { Department } from '../types';

// Mock data - replace with actual API calls
const mockDepartments: Department[] = [
  {
    id: '1',
    name: 'Engineering',
    description: 'Software development and technical operations',
    managerId: '1',
    employeeCount: 45,
    budget: 2500000,
  },
  {
    id: '2',
    name: 'Sales',
    description: 'Revenue generation and client acquisition',
    managerId: '2',
    employeeCount: 32,
    budget: 1800000,
  },
  {
    id: '3',
    name: 'Marketing',
    description: 'Brand promotion and market research',
    managerId: '3',
    employeeCount: 28,
    budget: 1200000,
  },
  {
    id: '4',
    name: 'Human Resources',
    description: 'Employee management and organizational development',
    managerId: '4',
    employeeCount: 15,
    budget: 800000,
  },
  {
    id: '5',
    name: 'Finance',
    description: 'Financial planning and accounting operations',
    managerId: '5',
    employeeCount: 25,
    budget: 1000000,
  },
];

let departments = [...mockDepartments];

export const departmentService = {
  // Get all departments
  getAllDepartments: async (): Promise<Department[]> => {
    return new Promise((resolve) => {
      setTimeout(() => resolve([...departments]), 500);
    });
  },

  // Get department by ID
  getDepartmentById: async (id: string): Promise<Department | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const department = departments.find(dept => dept.id === id);
        resolve(department || null);
      }, 300);
    });
  },

  // Create new department
  createDepartment: async (departmentData: Omit<Department, 'id'>): Promise<Department> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const newDepartment: Department = {
          ...departmentData,
          id: Date.now().toString(),
        };
        departments.push(newDepartment);
        resolve(newDepartment);
      }, 500);
    });
  },

  // Update department
  updateDepartment: async (id: string, departmentData: Partial<Department>): Promise<Department | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = departments.findIndex(dept => dept.id === id);
        if (index !== -1) {
          departments[index] = { ...departments[index], ...departmentData };
          resolve(departments[index]);
        } else {
          resolve(null);
        }
      }, 500);
    });
  },

  // Delete department
  deleteDepartment: async (id: string): Promise<boolean> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = departments.findIndex(dept => dept.id === id);
        if (index !== -1) {
          departments.splice(index, 1);
          resolve(true);
        } else {
          resolve(false);
        }
      }, 300);
    });
  },

  // Get department statistics
  getDepartmentStats: async (): Promise<{
    totalDepartments: number;
    totalEmployees: number;
    totalBudget: number;
    averageBudgetPerDepartment: number;
    averageEmployeesPerDepartment: number;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const totalEmployees = departments.reduce((sum, dept) => sum + dept.employeeCount, 0);
        const totalBudget = departments.reduce((sum, dept) => sum + dept.budget, 0);
        resolve({
          totalDepartments: departments.length,
          totalEmployees,
          totalBudget,
          averageBudgetPerDepartment: totalBudget / departments.length,
          averageEmployeesPerDepartment: totalEmployees / departments.length,
        });
      }, 300);
    });
  },

  // Get departments by budget range
  getDepartmentsByBudgetRange: async (minBudget: number, maxBudget: number): Promise<Department[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = departments.filter(dept => 
          dept.budget >= minBudget && dept.budget <= maxBudget
        );
        resolve(filtered);
      }, 300);
    });
  },

  // Update department employee count
  updateDepartmentEmployeeCount: async (id: string, count: number): Promise<boolean> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = departments.findIndex(dept => dept.id === id);
        if (index !== -1) {
          departments[index].employeeCount = count;
          resolve(true);
        } else {
          resolve(false);
        }
      }, 300);
    });
  },

  // Assign manager to department
  assignManager: async (departmentId: string, managerId: string): Promise<boolean> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = departments.findIndex(dept => dept.id === departmentId);
        if (index !== -1) {
          departments[index].managerId = managerId;
          resolve(true);
        } else {
          resolve(false);
        }
      }, 300);
    });
  },

  // Search departments
  searchDepartments: async (query: string): Promise<Department[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = departments.filter(dept =>
          dept.name.toLowerCase().includes(query.toLowerCase()) ||
          dept.description.toLowerCase().includes(query.toLowerCase())
        );
        resolve(filtered);
      }, 300);
    });
  }
};