import { DashboardStats } from '../types';
import { employeeService } from './employeeService';
import { attendanceService } from './attendanceService';
import { leaveService } from './leaveService';
import { payrollService } from './payrollService';
import { departmentService } from './departmentService';
import { format, subDays, subMonths } from 'date-fns';

export interface DashboardMetrics {
  overview: DashboardStats;
  trends: {
    employeeGrowth: Array<{
      month: string;
      employees: number;
      hires: number;
      terminations: number;
    }>;
    attendanceTrend: Array<{
      date: string;
      rate: number;
      present: number;
      absent: number;
    }>;
    leaveTrend: Array<{
      month: string;
      requests: number;
      approved: number;
      rejected: number;
    }>;
    payrollTrend: Array<{
      month: string;
      amount: number;
      employees: number;
    }>;
  };
  alerts: Array<{
    type: 'warning' | 'error' | 'info';
    message: string;
    actionUrl?: string;
  }>;
  upcomingEvents: Array<{
    type: 'birthday' | 'anniversary' | 'leave' | 'meeting';
    title: string;
    date: string;
    description?: string;
  }>;
  quickStats: {
    todayBirthdays: number;
    upcomingLeaves: number;
    pendingApprovals: number;
    overdueDocuments: number;
  };
}

// Mock trend data generators
const generateEmployeeGrowthTrend = () => [
  { month: 'Jan', employees: 120, hires: 5, terminations: 2 },
  { month: 'Feb', employees: 125, hires: 7, terminations: 2 },
  { month: 'Mar', employees: 130, hires: 8, terminations: 3 },
  { month: 'Apr', employees: 135, hires: 6, terminations: 1 },
  { month: 'May', employees: 140, hires: 7, terminations: 2 },
  { month: 'Jun', employees: 145, hires: 8, terminations: 3 },
];

const generateAttendanceTrend = () => {
  const trends = [];
  for (let i = 6; i >= 0; i--) {
    const date = subDays(new Date(), i);
    trends.push({
      date: format(date, 'MMM dd'),
      rate: Math.random() * 10 + 88, // 88-98% range
      present: Math.floor(Math.random() * 10) + 130,
      absent: Math.floor(Math.random() * 8) + 5,
    });
  }
  return trends;
};

const generateLeaveTrend = () => [
  { month: 'Jan', requests: 18, approved: 16, rejected: 2 },
  { month: 'Feb', requests: 22, approved: 20, rejected: 2 },
  { month: 'Mar', requests: 28, approved: 25, rejected: 3 },
  { month: 'Apr', requests: 25, approved: 23, rejected: 2 },
  { month: 'May', requests: 32, approved: 29, rejected: 3 },
  { month: 'Jun', requests: 31, approved: 28, rejected: 3 },
];

const generatePayrollTrend = () => [
  { month: 'Jan', amount: 6200000, employees: 120 },
  { month: 'Feb', amount: 6350000, employees: 125 },
  { month: 'Mar', amount: 6500000, employees: 130 },
  { month: 'Apr', amount: 6650000, employees: 135 },
  { month: 'May', amount: 6700000, employees: 140 },
  { month: 'Jun', amount: 6750000, employees: 145 },
];

export const dashboardService = {
  // Get comprehensive dashboard data
  getDashboardMetrics: async (userId?: string): Promise<DashboardMetrics> => {
    return new Promise(async (resolve) => {
      setTimeout(async () => {
        // Simulate fetching data from various services
        const overview: DashboardStats = {
          totalEmployees: 145,
          activeEmployees: 138,
          totalDepartments: 8,
          pendingLeaves: 12,
          todayAttendance: 134,
          monthlyPayroll: 6750000,
        };

        const alerts = [
          {
            type: 'warning' as const,
            message: '12 leave requests pending approval',
            actionUrl: '/leaves?filter=pending'
          },
          {
            type: 'info' as const,
            message: 'Monthly payroll ready for processing',
            actionUrl: '/payroll?status=processed'
          },
          {
            type: 'error' as const,
            message: '3 employees have exceeded overtime limits',
            actionUrl: '/attendance?filter=overtime'
          }
        ];

        const upcomingEvents = [
          {
            type: 'birthday' as const,
            title: 'John Doe\'s Birthday',
            date: format(new Date(), 'yyyy-MM-dd'),
            description: 'Engineering Team'
          },
          {
            type: 'anniversary' as const,
            title: 'Sarah Johnson - 2 Year Anniversary',
            date: format(subDays(new Date(), -2), 'yyyy-MM-dd'),
            description: 'Sales Department'
          },
          {
            type: 'leave' as const,
            title: 'Michael Brown - Vacation',
            date: format(subDays(new Date(), -5), 'yyyy-MM-dd'),
            description: '5 days vacation leave'
          },
          {
            type: 'meeting' as const,
            title: 'Monthly HR Review',
            date: format(subDays(new Date(), -7), 'yyyy-MM-dd'),
            description: 'Department heads meeting'
          }
        ];

        const quickStats = {
          todayBirthdays: 2,
          upcomingLeaves: 8,
          pendingApprovals: 15,
          overdueDocuments: 5,
        };

        const metrics: DashboardMetrics = {
          overview,
          trends: {
            employeeGrowth: generateEmployeeGrowthTrend(),
            attendanceTrend: generateAttendanceTrend(),
            leaveTrend: generateLeaveTrend(),
            payrollTrend: generatePayrollTrend(),
          },
          alerts,
          upcomingEvents,
          quickStats,
        };

        resolve(metrics);
      }, 800);
    });
  },

  // Get real-time statistics
  getRealTimeStats: async (): Promise<{
    currentlyPresent: number;
    lateToday: number;
    onLeaveToday: number;
    pendingApprovals: {
      leaves: number;
      expenses: number;
      timesheets: number;
    };
    recentActivities: Array<{
      id: string;
      activity: string;
      user: string;
      timestamp: string;
    }>;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const stats = {
          currentlyPresent: 134,
          lateToday: 7,
          onLeaveToday: 4,
          pendingApprovals: {
            leaves: 12,
            expenses: 8,
            timesheets: 23,
          },
          recentActivities: [
            {
              id: '1',
              activity: 'Sarah Johnson submitted leave request',
              user: 'Sarah Johnson',
              timestamp: format(new Date(Date.now() - 15 * 60 * 1000), 'HH:mm')
            },
            {
              id: '2',
              activity: 'Michael Brown clocked in',
              user: 'Michael Brown',
              timestamp: format(new Date(Date.now() - 30 * 60 * 1000), 'HH:mm')
            },
            {
              id: '3',
              activity: 'Payroll processed for Engineering team',
              user: 'HR System',
              timestamp: format(new Date(Date.now() - 45 * 60 * 1000), 'HH:mm')
            },
            {
              id: '4',
              activity: 'Emily Davis approved leave request',
              user: 'Emily Davis',
              timestamp: format(new Date(Date.now() - 60 * 60 * 1000), 'HH:mm')
            }
          ],
        };

        resolve(stats);
      }, 500);
    });
  },

  // Get department overview
  getDepartmentOverview: async (): Promise<Array<{
    id: string;
    name: string;
    employeeCount: number;
    attendanceRate: number;
    avgSalary: number;
    pendingLeaves: number;
    budget: number;
    utilization: number;
    performance: number;
  }>> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const overview = [
          {
            id: '1',
            name: 'Engineering',
            employeeCount: 45,
            attendanceRate: 94.2,
            avgSalary: 55556,
            pendingLeaves: 5,
            budget: 2500000,
            utilization: 95.2,
            performance: 92.8
          },
          {
            id: '2',
            name: 'Sales',
            employeeCount: 32,
            attendanceRate: 89.8,
            avgSalary: 56250,
            pendingLeaves: 3,
            budget: 1800000,
            utilization: 87.3,
            performance: 88.5
          },
          {
            id: '3',
            name: 'Marketing',
            employeeCount: 28,
            attendanceRate: 91.5,
            avgSalary: 42857,
            pendingLeaves: 2,
            budget: 1200000,
            utilization: 91.8,
            performance: 90.2
          },
          {
            id: '4',
            name: 'HR',
            employeeCount: 15,
            attendanceRate: 96.7,
            avgSalary: 53333,
            pendingLeaves: 1,
            budget: 800000,
            utilization: 98.1,
            performance: 94.5
          },
          {
            id: '5',
            name: 'Finance',
            employeeCount: 25,
            attendanceRate: 93.2,
            avgSalary: 40000,
            pendingLeaves: 1,
            budget: 1000000,
            utilization: 93.5,
            performance: 91.8
          }
        ];

        resolve(overview);
      }, 600);
    });
  },

  // Get financial summary
  getFinancialSummary: async (): Promise<{
    currentMonth: {
      totalPayroll: number;
      totalAllowances: number;
      totalDeductions: number;
      netPayable: number;
    };
    yearToDate: {
      totalPayroll: number;
      averageMonthly: number;
      projectedYearly: number;
    };
    comparison: {
      lastMonth: {
        change: number;
        percentage: number;
      };
      lastYear: {
        change: number;
        percentage: number;
      };
    };
    budgetUtilization: Array<{
      department: string;
      allocated: number;
      spent: number;
      remaining: number;
      utilization: number;
    }>;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const summary = {
          currentMonth: {
            totalPayroll: 6750000,
            totalAllowances: 675000,
            totalDeductions: 337500,
            netPayable: 6210000,
          },
          yearToDate: {
            totalPayroll: 38250000,
            averageMonthly: 6375000,
            projectedYearly: 81000000,
          },
          comparison: {
            lastMonth: {
              change: 50000,
              percentage: 0.75,
            },
            lastYear: {
              change: 3250000,
              percentage: 5.05,
            },
          },
          budgetUtilization: [
            { department: 'Engineering', allocated: 3000000, spent: 2500000, remaining: 500000, utilization: 83.3 },
            { department: 'Sales', allocated: 2200000, spent: 1800000, remaining: 400000, utilization: 81.8 },
            { department: 'Marketing', allocated: 1500000, spent: 1200000, remaining: 300000, utilization: 80.0 },
            { department: 'HR', allocated: 1000000, spent: 800000, remaining: 200000, utilization: 80.0 },
            { department: 'Finance', allocated: 1200000, spent: 1000000, remaining: 200000, utilization: 83.3 },
          ]
        };

        resolve(summary);
      }, 700);
    });
  },

  // Get employee lifecycle metrics
  getEmployeeLifecycleMetrics: async (): Promise<{
    hiring: {
      monthlyHires: number;
      averageTimeToHire: number; // days
      sourceBreakdown: Array<{ source: string; count: number }>;
    };
    retention: {
      overallRetentionRate: number;
      avgTenure: number; // years
      turnoverRate: number;
      exitReasons: Array<{ reason: string; count: number }>;
    };
    onboarding: {
      activeOnboarding: number;
      completionRate: number;
      averageDaysToComplete: number;
    };
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const metrics = {
          hiring: {
            monthlyHires: 8,
            averageTimeToHire: 21,
            sourceBreakdown: [
              { source: 'Job Boards', count: 35 },
              { source: 'Referrals', count: 28 },
              { source: 'LinkedIn', count: 22 },
              { source: 'Campus Hiring', count: 15 }
            ]
          },
          retention: {
            overallRetentionRate: 94.2,
            avgTenure: 2.8,
            turnoverRate: 5.8,
            exitReasons: [
              { reason: 'Better Opportunity', count: 12 },
              { reason: 'Career Growth', count: 8 },
              { reason: 'Work-Life Balance', count: 5 },
              { reason: 'Compensation', count: 4 },
              { reason: 'Personal Reasons', count: 3 }
            ]
          },
          onboarding: {
            activeOnboarding: 3,
            completionRate: 87.5,
            averageDaysToComplete: 14,
          }
        };

        resolve(metrics);
      }, 500);
    });
  },

  // Get performance insights
  getPerformanceInsights: async (): Promise<{
    overallScore: number;
    departmentPerformance: Array<{
      department: string;
      score: number;
      trend: 'up' | 'down' | 'stable';
    }>;
    topPerformers: Array<{
      employeeId: string;
      name: string;
      department: string;
      score: number;
    }>;
    improvementAreas: Array<{
      area: string;
      impact: 'high' | 'medium' | 'low';
      recommendation: string;
    }>;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const insights = {
          overallScore: 87.5,
          departmentPerformance: [
            { department: 'Engineering', score: 91.2, trend: 'up' as const },
            { department: 'HR', score: 89.1, trend: 'stable' as const },
            { department: 'Finance', score: 88.7, trend: 'up' as const },
            { department: 'Marketing', score: 87.4, trend: 'stable' as const },
            { department: 'Sales', score: 85.8, trend: 'down' as const },
          ],
          topPerformers: [
            { employeeId: '1', name: 'John Doe', department: 'Engineering', score: 98.5 },
            { employeeId: '2', name: 'Sarah Johnson', department: 'Sales', score: 96.2 },
            { employeeId: '3', name: 'Michael Brown', department: 'Marketing', score: 94.8 },
            { employeeId: '4', name: 'Emily Davis', department: 'HR', score: 94.5 },
            { employeeId: '5', name: 'David Wilson', department: 'Finance', score: 91.8 },
          ],
          improvementAreas: [
            {
              area: 'Sales Team Performance',
              impact: 'high' as const,
              recommendation: 'Implement additional training and mentorship programs'
            },
            {
              area: 'Attendance Rate',
              impact: 'medium' as const,
              recommendation: 'Consider flexible working arrangements'
            },
            {
              area: 'Leave Processing Time',
              impact: 'medium' as const,
              recommendation: 'Automate leave approval workflow'
            }
          ]
        };

        resolve(insights);
      }, 600);
    });
  },

  // Get cost analysis
  getCostAnalysis: async (): Promise<{
    totalCosts: {
      salaries: number;
      benefits: number;
      training: number;
      overhead: number;
      total: number;
    };
    costPerEmployee: number;
    departmentCosts: Array<{
      department: string;
      totalCost: number;
      costPerEmployee: number;
      budgetUtilization: number;
    }>;
    trends: Array<{
      month: string;
      totalCost: number;
      change: number;
    }>;
    projections: {
      nextMonth: number;
      nextQuarter: number;
      yearEnd: number;
    };
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const analysis = {
          totalCosts: {
            salaries: 6750000,
            benefits: 1012500, // 15% of salaries
            training: 135000, // 2% of salaries
            overhead: 337500, // 5% of salaries
            total: 8235000,
          },
          costPerEmployee: 56793,
          departmentCosts: [
            { department: 'Engineering', totalCost: 3000000, costPerEmployee: 66667, budgetUtilization: 83.3 },
            { department: 'Sales', totalCost: 2160000, costPerEmployee: 67500, budgetUtilization: 81.8 },
            { department: 'Marketing', totalCost: 1440000, costPerEmployee: 51429, budgetUtilization: 80.0 },
            { department: 'Finance', totalCost: 1200000, costPerEmployee: 48000, budgetUtilization: 83.3 },
            { department: 'HR', totalCost: 960000, costPerEmployee: 64000, budgetUtilization: 80.0 },
          ],
          trends: [
            { month: 'Jan', totalCost: 7440000, change: 0 },
            { month: 'Feb', totalCost: 7627500, change: 187500 },
            { month: 'Mar', totalCost: 7815000, change: 187500 },
            { month: 'Apr', totalCost: 8002500, change: 187500 },
            { month: 'May', totalCost: 8120000, change: 117500 },
            { month: 'Jun', totalCost: 8235000, change: 115000 },
          ],
          projections: {
            nextMonth: 8350000,
            nextQuarter: 25500000,
            yearEnd: 98820000,
          }
        };

        resolve(analysis);
      }, 500);
    });
  },

  // Get compliance metrics
  getComplianceMetrics: async (): Promise<{
    documentCompliance: {
      rate: number;
      missing: Array<{ employee: string; documents: string[] }>;
    };
    trainingCompliance: {
      rate: number;
      overdue: Array<{ employee: string; courses: string[] }>;
    };
    policyAcknowledgment: {
      rate: number;
      pending: Array<{ employee: string; policies: string[] }>;
    };
    certificationStatus: {
      active: number;
      expiring: number;
      expired: number;
    };
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const metrics = {
          documentCompliance: {
            rate: 92.4,
            missing: [
              { employee: 'John Doe', documents: ['Emergency Contact Form'] },
              { employee: 'Alex Wilson', documents: ['Tax Forms', 'Direct Deposit'] },
            ]
          },
          trainingCompliance: {
            rate: 87.6,
            overdue: [
              { employee: 'Sarah Johnson', courses: ['Safety Training', 'Data Protection'] },
              { employee: 'Lisa Chen', courses: ['Compliance Training'] },
            ]
          },
          policyAcknowledgment: {
            rate: 95.2,
            pending: [
              { employee: 'Michael Brown', policies: ['Remote Work Policy'] },
              { employee: 'David Wilson', policies: ['Updated IT Policy'] },
            ]
          },
          certificationStatus: {
            active: 89,
            expiring: 12,
            expired: 3,
          }
        };

        resolve(metrics);
      }, 500);
    });
  },

  // Refresh dashboard cache
  refreshCache: async (): Promise<{ success: boolean; lastUpdated: string }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        resolve({
          success: true,
          lastUpdated: new Date().toISOString(),
        });
      }, 1000);
    });
  },

  // Get widget configuration
  getWidgetConfiguration: async (userId: string): Promise<{
    layout: Array<{
      id: string;
      name: string;
      size: 'small' | 'medium' | 'large';
      position: { row: number; col: number };
      visible: boolean;
    }>;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        // Mock widget configuration
        const layout = [
          { id: 'stats-overview', name: 'Statistics Overview', size: 'large' as const, position: { row: 0, col: 0 }, visible: true },
          { id: 'attendance-chart', name: 'Attendance Chart', size: 'medium' as const, position: { row: 1, col: 0 }, visible: true },
          { id: 'department-pie', name: 'Department Distribution', size: 'medium' as const, position: { row: 1, col: 1 }, visible: true },
          { id: 'recent-activities', name: 'Recent Activities', size: 'small' as const, position: { row: 2, col: 0 }, visible: true },
          { id: 'upcoming-events', name: 'Upcoming Events', size: 'small' as const, position: { row: 2, col: 1 }, visible: true },
        ];

        resolve({ layout });
      }, 300);
    });
  },

  // Update widget configuration
  updateWidgetConfiguration: async (userId: string, layout: any): Promise<{ success: boolean }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        // Mock save configuration
        resolve({ success: true });
      }, 300);
    });
  },
};