import { Attendance } from '../types';
import { format, startOfWeek, endOfWeek, startOfMonth, endOfMonth, parseISO } from 'date-fns';

// Mock data - replace with actual API calls
const mockAttendance: Attendance[] = [
  {
    id: '1',
    employeeId: '1',
    date: '2023-06-19',
    checkIn: '09:00',
    checkOut: '17:30',
    status: 'present',
    workingHours: 8,
    overtime: 0.5,
  },
  {
    id: '2',
    employeeId: '2',
    date: '2023-06-19',
    checkIn: '09:15',
    checkOut: '17:45',
    status: 'late',
    workingHours: 8,
    overtime: 0.5,
  },
  {
    id: '3',
    employeeId: '3',
    date: '2023-06-19',
    checkIn: '09:00',
    checkOut: '13:00',
    status: 'half-day',
    workingHours: 4,
    overtime: 0,
  },
  {
    id: '4',
    employeeId: '4',
    date: '2023-06-19',
    status: 'absent',
    workingHours: 0,
    overtime: 0,
  },
  {
    id: '5',
    employeeId: '5',
    date: '2023-06-19',
    checkIn: '09:00',
    checkOut: '17:30',
    status: 'present',
    workingHours: 8,
    overtime: 0.5,
  },
];

let attendanceRecords = [...mockAttendance];

export const attendanceService = {
  // Get all attendance records
  getAllAttendance: async (): Promise<Attendance[]> => {
    return new Promise((resolve) => {
      setTimeout(() => resolve([...attendanceRecords]), 500);
    });
  },

  // Get attendance by date
  getAttendanceByDate: async (date: string): Promise<Attendance[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = attendanceRecords.filter(record => record.date === date);
        resolve(filtered);
      }, 300);
    });
  },

  // Get attendance by employee ID
  getAttendanceByEmployeeId: async (employeeId: string): Promise<Attendance[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = attendanceRecords.filter(record => record.employeeId === employeeId);
        resolve(filtered);
      }, 300);
    });
  },

  // Get attendance by date range
  getAttendanceByDateRange: async (startDate: string, endDate: string): Promise<Attendance[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = attendanceRecords.filter(record => {
          const recordDate = parseISO(record.date);
          const start = parseISO(startDate);
          const end = parseISO(endDate);
          return recordDate >= start && recordDate <= end;
        });
        resolve(filtered);
      }, 300);
    });
  },

  // Create attendance record
  createAttendance: async (attendanceData: Omit<Attendance, 'id'>): Promise<Attendance> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const newAttendance: Attendance = {
          ...attendanceData,
          id: Date.now().toString(),
        };
        attendanceRecords.push(newAttendance);
        resolve(newAttendance);
      }, 500);
    });
  },

  // Update attendance record
  updateAttendance: async (id: string, attendanceData: Partial<Attendance>): Promise<Attendance | null> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = attendanceRecords.findIndex(record => record.id === id);
        if (index !== -1) {
          attendanceRecords[index] = { ...attendanceRecords[index], ...attendanceData };
          resolve(attendanceRecords[index]);
        } else {
          resolve(null);
        }
      }, 500);
    });
  },

  // Delete attendance record
  deleteAttendance: async (id: string): Promise<boolean> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = attendanceRecords.findIndex(record => record.id === id);
        if (index !== -1) {
          attendanceRecords.splice(index, 1);
          resolve(true);
        } else {
          resolve(false);
        }
      }, 300);
    });
  },

  // Clock in
  clockIn: async (employeeId: string, time: string, date?: string): Promise<Attendance> => {
    const attendanceDate = date || format(new Date(), 'yyyy-MM-dd');
    const existingRecord = attendanceRecords.find(
      record => record.employeeId === employeeId && record.date === attendanceDate
    );

    return new Promise((resolve) => {
      setTimeout(() => {
        if (existingRecord) {
          existingRecord.checkIn = time;
          existingRecord.status = 'present';
          resolve(existingRecord);
        } else {
          const newRecord: Attendance = {
            id: Date.now().toString(),
            employeeId,
            date: attendanceDate,
            checkIn: time,
            status: 'present',
            workingHours: 0,
            overtime: 0,
          };
          attendanceRecords.push(newRecord);
          resolve(newRecord);
        }
      }, 500);
    });
  },

  // Clock out
  clockOut: async (employeeId: string, time: string, date?: string): Promise<Attendance | null> => {
    const attendanceDate = date || format(new Date(), 'yyyy-MM-dd');
    
    return new Promise((resolve) => {
      setTimeout(() => {
        const record = attendanceRecords.find(
          record => record.employeeId === employeeId && record.date === attendanceDate
        );

        if (record && record.checkIn) {
          record.checkOut = time;
          
          // Calculate working hours
          const checkInTime = new Date(`${attendanceDate}T${record.checkIn}`);
          const checkOutTime = new Date(`${attendanceDate}T${time}`);
          const diffInHours = (checkOutTime.getTime() - checkInTime.getTime()) / (1000 * 60 * 60);
          
          record.workingHours = Math.round(diffInHours * 2) / 2; // Round to nearest 0.5
          record.overtime = Math.max(0, record.workingHours - 8); // Standard 8-hour workday
          
          resolve(record);
        } else {
          resolve(null);
        }
      }, 500);
    });
  },

  // Get attendance statistics
  getAttendanceStats: async (date?: string): Promise<{
    total: number;
    present: number;
    absent: number;
    late: number;
    halfDay: number;
    attendanceRate: number;
  }> => {
    const targetDate = date || format(new Date(), 'yyyy-MM-dd');
    
    return new Promise((resolve) => {
      setTimeout(() => {
        const dayRecords = attendanceRecords.filter(record => record.date === targetDate);
        const stats = {
          total: dayRecords.length,
          present: dayRecords.filter(r => r.status === 'present').length,
          absent: dayRecords.filter(r => r.status === 'absent').length,
          late: dayRecords.filter(r => r.status === 'late').length,
          halfDay: dayRecords.filter(r => r.status === 'half-day').length,
          attendanceRate: 0,
        };
        
        stats.attendanceRate = stats.total > 0 
          ? Math.round(((stats.present + stats.late + stats.halfDay) / stats.total) * 100)
          : 0;
        
        resolve(stats);
      }, 300);
    });
  },

  // Get weekly attendance summary
  getWeeklyAttendance: async (date?: string): Promise<{
    weekStart: string;
    weekEnd: string;
    dailyStats: Array<{
      date: string;
      present: number;
      absent: number;
      late: number;
      halfDay: number;
    }>;
  }> => {
    const targetDate = date ? parseISO(date) : new Date();
    const weekStart = startOfWeek(targetDate);
    const weekEnd = endOfWeek(targetDate);

    return new Promise((resolve) => {
      setTimeout(() => {
        const weekRecords = attendanceRecords.filter(record => {
          const recordDate = parseISO(record.date);
          return recordDate >= weekStart && recordDate <= weekEnd;
        });

        const dailyStats: { [key: string]: any } = {};
        
        weekRecords.forEach(record => {
          if (!dailyStats[record.date]) {
            dailyStats[record.date] = {
              date: record.date,
              present: 0,
              absent: 0,
              late: 0,
              halfDay: 0,
            };
          }
          dailyStats[record.date][record.status === 'half-day' ? 'halfDay' : record.status]++;
        });

        resolve({
          weekStart: format(weekStart, 'yyyy-MM-dd'),
          weekEnd: format(weekEnd, 'yyyy-MM-dd'),
          dailyStats: Object.values(dailyStats),
        });
      }, 500);
    });
  },

  // Get monthly attendance summary
  getMonthlyAttendance: async (year: number, month: number): Promise<{
    month: string;
    totalWorkingDays: number;
    totalAttendanceRecords: number;
    monthlyStats: {
      present: number;
      absent: number;
      late: number;
      halfDay: number;
      averageWorkingHours: number;
      totalOvertimeHours: number;
    };
  }> => {
    const monthStart = startOfMonth(new Date(year, month - 1));
    const monthEnd = endOfMonth(new Date(year, month - 1));

    return new Promise((resolve) => {
      setTimeout(() => {
        const monthRecords = attendanceRecords.filter(record => {
          const recordDate = parseISO(record.date);
          return recordDate >= monthStart && recordDate <= monthEnd;
        });

        const stats = {
          present: monthRecords.filter(r => r.status === 'present').length,
          absent: monthRecords.filter(r => r.status === 'absent').length,
          late: monthRecords.filter(r => r.status === 'late').length,
          halfDay: monthRecords.filter(r => r.status === 'half-day').length,
          averageWorkingHours: 0,
          totalOvertimeHours: 0,
        };

        const workingRecords = monthRecords.filter(r => r.workingHours && r.workingHours > 0);
        if (workingRecords.length > 0) {
          stats.averageWorkingHours = workingRecords.reduce((sum, r) => sum + (r.workingHours || 0), 0) / workingRecords.length;
          stats.totalOvertimeHours = monthRecords.reduce((sum, r) => sum + (r.overtime || 0), 0);
        }

        resolve({
          month: format(monthStart, 'MMMM yyyy'),
          totalWorkingDays: 22, // Assumed working days per month
          totalAttendanceRecords: monthRecords.length,
          monthlyStats: stats,
        });
      }, 500);
    });
  },

  // Mark employee as absent
  markAbsent: async (employeeId: string, date: string, reason?: string): Promise<Attendance> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const existingRecord = attendanceRecords.find(
          record => record.employeeId === employeeId && record.date === date
        );

        if (existingRecord) {
          existingRecord.status = 'absent';
          existingRecord.workingHours = 0;
          existingRecord.overtime = 0;
          existingRecord.notes = reason;
          resolve(existingRecord);
        } else {
          const newRecord: Attendance = {
            id: Date.now().toString(),
            employeeId,
            date,
            status: 'absent',
            workingHours: 0,
            overtime: 0,
            notes: reason,
          };
          attendanceRecords.push(newRecord);
          resolve(newRecord);
        }
      }, 500);
    });
  },

  // Bulk attendance operations
  bulkCreateAttendance: async (attendanceList: Omit<Attendance, 'id'>[]): Promise<Attendance[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const newRecords = attendanceList.map(data => ({
          ...data,
          id: `${Date.now()}_${Math.random()}`,
        }));
        attendanceRecords.push(...newRecords);
        resolve(newRecords);
      }, 1000);
    });
  },

  // Get employee attendance summary
  getEmployeeAttendanceSummary: async (employeeId: string, startDate: string, endDate: string): Promise<{
    employeeId: string;
    period: string;
    totalDays: number;
    presentDays: number;
    absentDays: number;
    lateDays: number;
    halfDays: number;
    totalWorkingHours: number;
    totalOvertimeHours: number;
    attendanceRate: number;
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const records = attendanceRecords.filter(record => 
          record.employeeId === employeeId &&
          record.date >= startDate &&
          record.date <= endDate
        );

        const summary = {
          employeeId,
          period: `${startDate} to ${endDate}`,
          totalDays: records.length,
          presentDays: records.filter(r => r.status === 'present').length,
          absentDays: records.filter(r => r.status === 'absent').length,
          lateDays: records.filter(r => r.status === 'late').length,
          halfDays: records.filter(r => r.status === 'half-day').length,
          totalWorkingHours: records.reduce((sum, r) => sum + (r.workingHours || 0), 0),
          totalOvertimeHours: records.reduce((sum, r) => sum + (r.overtime || 0), 0),
          attendanceRate: 0,
        };

        summary.attendanceRate = summary.totalDays > 0
          ? Math.round(((summary.presentDays + summary.lateDays + summary.halfDays) / summary.totalDays) * 100)
          : 0;

        resolve(summary);
      }, 500);
    });
  }
};