import React, { useState, useEffect } from 'react';
import {
  Box,
  Card,
  CardContent,
  Typography,
  Button,
  Tabs,
  Tab,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  Paper,
  Chip,
  IconButton,
  Menu,
  MenuItem,
  Dialog,
  DialogActions,
  DialogContent,
  DialogContentText,
  DialogTitle,
  Fab,
  TextField,
  FormControl,
  InputLabel,
  Select,
  MenuItem as SelectMenuItem
} from '@mui/material';
import {
  Add,
  Work,
  People,
  Schedule,
  Assignment,
  MoreVert,
  Edit,
  Delete,
  Visibility,
  FilterList,
  Search,
  Business,
  TrendingUp
} from '@mui/icons-material';
import { JobPosting, Candidate, Interview, OnboardingProcess, RecruitmentMetrics, JobPostingRequest } from '../types/recruitment';
import { recruitmentService } from '../services/recruitmentService';
import { LoaderOverlay } from '../components/Common/LoaderOverlay';
import { JobPostingDialog } from '../components/Recruitment/JobPostingDialog';

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

function TabPanel(props: TabPanelProps) {
  const { children, value, index, ...other } = props;
  return (
    <div
      role="tabpanel"
      hidden={value !== index}
      id={`recruitment-tabpanel-${index}`}
      aria-labelledby={`recruitment-tab-${index}`}
      {...other}
    >
      {value === index && <Box>{children}</Box>}
    </div>
  );
}

export const Recruitment: React.FC = () => {
  const [loading, setLoading] = useState(true);
  const [tabValue, setTabValue] = useState(0);
  const [jobPostings, setJobPostings] = useState<JobPosting[]>([]);
  const [candidates, setCandidates] = useState<Candidate[]>([]);
  const [interviews, setInterviews] = useState<Interview[]>([]);
  const [onboardingProcesses, setOnboardingProcesses] = useState<OnboardingProcess[]>([]);
  const [metrics, setMetrics] = useState<RecruitmentMetrics | null>(null);
  
  const [jobDialogOpen, setJobDialogOpen] = useState(false);
  const [editingJob, setEditingJob] = useState<JobPosting | undefined>();
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [deletingJobId, setDeletingJobId] = useState<string>('');
  
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('');
  const [departmentFilter, setDepartmentFilter] = useState('');
  
  const [anchorEl, setAnchorEl] = useState<null | HTMLElement>(null);
  const [selectedJobId, setSelectedJobId] = useState<string>('');

  const loadData = async () => {
    setLoading(true);
    try {
      const [jobsRes, candidatesRes, interviewsRes, onboardingRes, metricsRes] = await Promise.all([
        recruitmentService.getJobPostings(),
        recruitmentService.getCandidates(),
        recruitmentService.getInterviews(),
        recruitmentService.getOnboardingProcesses(),
        recruitmentService.getRecruitmentMetrics()
      ]);
      
      setJobPostings(jobsRes);
      setCandidates(candidatesRes);
      setInterviews(interviewsRes);
      setOnboardingProcesses(onboardingRes);
      setMetrics(metricsRes);
    } catch (error) {
      console.error('Failed to load recruitment data:', error);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    loadData();
  }, []);

  const handleTabChange = (event: React.SyntheticEvent, newValue: number) => {
    setTabValue(newValue);
  };

  const handleCreateJob = () => {
    setEditingJob(undefined);
    setJobDialogOpen(true);
  };

  const handleEditJob = (job: JobPosting) => {
    setEditingJob(job);
    setJobDialogOpen(true);
  };

  const handleDeleteJob = (jobId: string) => {
    setDeletingJobId(jobId);
    setDeleteDialogOpen(true);
  };

  const handleJobSubmit = async (data: JobPostingRequest) => {
    if (editingJob) {
      await recruitmentService.updateJobPosting(editingJob.id, data);
    } else {
      await recruitmentService.createJobPosting(data);
    }
    await loadData();
    setJobDialogOpen(false);
  };

  const confirmDelete = async () => {
    if (deletingJobId) {
      await recruitmentService.deleteJobPosting(deletingJobId);
      await loadData();
      setDeleteDialogOpen(false);
      setDeletingJobId('');
    }
  };

  const handleMenuOpen = (event: React.MouseEvent<HTMLElement>, jobId: string) => {
    setAnchorEl(event.currentTarget);
    setSelectedJobId(jobId);
  };

  const handleMenuClose = () => {
    setAnchorEl(null);
    setSelectedJobId('');
  };

  const filteredJobPostings = jobPostings.filter(job => {
    const matchesSearch = job.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                         job.department.toLowerCase().includes(searchTerm.toLowerCase());
    const matchesStatus = !statusFilter || job.status === statusFilter;
    const matchesDepartment = !departmentFilter || job.department === departmentFilter;
    return matchesSearch && matchesStatus && matchesDepartment;
  });

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'Published': return 'success';
      case 'Draft': return 'warning';
      case 'Closed': return 'error';
      case 'On Hold': return 'info';
      default: return 'default';
    }
  };

  if (loading) {
    return <LoaderOverlay open={loading} />;
  }

  return (
    <Box sx={{ p: 3 }}>
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
        <Typography variant="h4" fontWeight="bold">
          Recruitment & Onboarding
        </Typography>
        <Button 
          variant="contained" 
          startIcon={<Add />}
          onClick={handleCreateJob}
          sx={{ display: { xs: 'none', sm: 'flex' } }}
        >
          New Job Posting
        </Button>
      </Box>

      {metrics && (
        <Box sx={{ display: 'grid', gridTemplateColumns: { xs: '1fr', sm: '1fr 1fr', md: 'repeat(4, 1fr)' }, gap: 3, mb: 4 }}>
          <Card>
            <CardContent sx={{ textAlign: 'center' }}>
              <Work sx={{ fontSize: 40, color: 'primary.main', mb: 1 }} />
              <Typography variant="h4" fontWeight="bold">{metrics.activeJobPostings}</Typography>
              <Typography variant="body2" color="text.secondary">Active Jobs</Typography>
            </CardContent>
          </Card>

          <Card>
            <CardContent sx={{ textAlign: 'center' }}>
              <People sx={{ fontSize: 40, color: 'success.main', mb: 1 }} />
              <Typography variant="h4" fontWeight="bold">{metrics.totalApplications}</Typography>
              <Typography variant="body2" color="text.secondary">Applications</Typography>
            </CardContent>
          </Card>

          <Card>
            <CardContent sx={{ textAlign: 'center' }}>
              <Schedule sx={{ fontSize: 40, color: 'warning.main', mb: 1 }} />
              <Typography variant="h4" fontWeight="bold">{metrics.interviewsScheduled}</Typography>
              <Typography variant="body2" color="text.secondary">Interviews</Typography>
            </CardContent>
          </Card>

          <Card>
            <CardContent sx={{ textAlign: 'center' }}>
              <TrendingUp sx={{ fontSize: 40, color: 'info.main', mb: 1 }} />
              <Typography variant="h4" fontWeight="bold">{metrics.averageTimeToHire}</Typography>
              <Typography variant="body2" color="text.secondary">Avg. Days to Hire</Typography>
            </CardContent>
          </Card>
        </Box>
      )}

      <Card>
        <Box sx={{ borderBottom: 1, borderColor: 'divider' }}>
          <Tabs value={tabValue} onChange={handleTabChange}>
            <Tab label="Job Postings" />
            <Tab label="Candidates" />
            <Tab label="Interviews" />
            <Tab label="Onboarding" />
          </Tabs>
        </Box>

        <TabPanel value={tabValue} index={0}>
          <CardContent>
            <Box sx={{ display: 'flex', gap: 2, mb: 3, flexWrap: 'wrap' }}>
              <TextField
                placeholder="Search jobs..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                size="small"
                InputProps={{
                  startAdornment: <Search sx={{ mr: 1, color: 'text.secondary' }} />
                }}
                sx={{ minWidth: 200 }}
              />
              
              <FormControl size="small" sx={{ minWidth: 120 }}>
                <InputLabel>Status</InputLabel>
                <Select
                  value={statusFilter}
                  label="Status"
                  onChange={(e) => setStatusFilter(e.target.value)}
                >
                  <SelectMenuItem value="">All</SelectMenuItem>
                  <SelectMenuItem value="Published">Published</SelectMenuItem>
                  <SelectMenuItem value="Draft">Draft</SelectMenuItem>
                  <SelectMenuItem value="Closed">Closed</SelectMenuItem>
                  <SelectMenuItem value="On Hold">On Hold</SelectMenuItem>
                </Select>
              </FormControl>
              
              <FormControl size="small" sx={{ minWidth: 150 }}>
                <InputLabel>Department</InputLabel>
                <Select
                  value={departmentFilter}
                  label="Department"
                  onChange={(e) => setDepartmentFilter(e.target.value)}
                >
                  <SelectMenuItem value="">All</SelectMenuItem>
                  <SelectMenuItem value="Engineering">Engineering</SelectMenuItem>
                  <SelectMenuItem value="Product">Product</SelectMenuItem>
                  <SelectMenuItem value="Sales">Sales</SelectMenuItem>
                  <SelectMenuItem value="Marketing">Marketing</SelectMenuItem>
                </Select>
              </FormControl>
            </Box>

            <TableContainer component={Paper} variant="outlined">
              <Table>
                <TableHead>
                  <TableRow>
                    <TableCell>Job Title</TableCell>
                    <TableCell>Department</TableCell>
                    <TableCell>Location</TableCell>
                    <TableCell>Type</TableCell>
                    <TableCell>Applications</TableCell>
                    <TableCell>Status</TableCell>
                    <TableCell>Actions</TableCell>
                  </TableRow>
                </TableHead>
                <TableBody>
                  {filteredJobPostings.map((job) => (
                    <TableRow key={job.id} hover>
                      <TableCell>
                        <Typography fontWeight="medium">{job.title}</Typography>
                        <Typography variant="caption" color="text.secondary">
                          Posted: {new Date(job.postedDate).toLocaleDateString()}
                        </Typography>
                      </TableCell>
                      <TableCell>{job.department}</TableCell>
                      <TableCell>{job.location}</TableCell>
                      <TableCell>{job.employmentType}</TableCell>
                      <TableCell>
                        <Chip label={job.applicationCount} size="small" variant="outlined" />
                      </TableCell>
                      <TableCell>
                        <Chip 
                          label={job.status} 
                          size="small" 
                          color={getStatusColor(job.status) as any}
                        />
                      </TableCell>
                      <TableCell>
                        <IconButton onClick={(e) => handleMenuOpen(e, job.id)}>
                          <MoreVert />
                        </IconButton>
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </TableContainer>

            {filteredJobPostings.length === 0 && (
              <Box sx={{ textAlign: 'center', py: 4 }}>
                <Business sx={{ fontSize: 60, color: 'text.secondary', mb: 2 }} />
                <Typography variant="h6" color="text.secondary" gutterBottom>
                  No job postings found
                </Typography>
                <Typography variant="body2" color="text.secondary">
                  Create your first job posting to start recruiting
                </Typography>
                <Button 
                  variant="contained" 
                  startIcon={<Add />}
                  onClick={handleCreateJob}
                  sx={{ mt: 2 }}
                >
                  Create Job Posting
                </Button>
              </Box>
            )}
          </CardContent>
        </TabPanel>

        <TabPanel value={tabValue} index={1}>
          <CardContent>
            <Typography variant="h6" gutterBottom>Candidate Database</Typography>
            <Typography variant="body2" color="text.secondary">
              Candidate management functionality will be implemented here.
            </Typography>
          </CardContent>
        </TabPanel>

        <TabPanel value={tabValue} index={2}>
          <CardContent>
            <Typography variant="h6" gutterBottom>Interview Scheduling</Typography>
            <Typography variant="body2" color="text.secondary">
              Interview scheduling functionality will be implemented here.
            </Typography>
          </CardContent>
        </TabPanel>

        <TabPanel value={tabValue} index={3}>
          <CardContent>
            <Typography variant="h6" gutterBottom>Onboarding Processes</Typography>
            <Typography variant="body2" color="text.secondary">
              Onboarding workflow functionality will be implemented here.
            </Typography>
          </CardContent>
        </TabPanel>
      </Card>

      <Fab
        color="primary"
        sx={{ position: 'fixed', bottom: 16, right: 16, display: { xs: 'flex', sm: 'none' } }}
        onClick={handleCreateJob}
      >
        <Add />
      </Fab>

      <Menu
        anchorEl={anchorEl}
        open={Boolean(anchorEl)}
        onClose={handleMenuClose}
      >
        <MenuItem onClick={() => {
          const job = jobPostings.find(j => j.id === selectedJobId);
          if (job) handleEditJob(job);
          handleMenuClose();
        }}>
          <Edit sx={{ mr: 1 }} /> Edit
        </MenuItem>
        <MenuItem onClick={() => {
          handleDeleteJob(selectedJobId);
          handleMenuClose();
        }}>
          <Delete sx={{ mr: 1 }} /> Delete
        </MenuItem>
      </Menu>

      <JobPostingDialog
        open={jobDialogOpen}
        onClose={() => setJobDialogOpen(false)}
        onSubmit={handleJobSubmit}
        editData={editingJob}
      />

      <Dialog open={deleteDialogOpen} onClose={() => setDeleteDialogOpen(false)}>
        <DialogTitle>Confirm Delete</DialogTitle>
        <DialogContent>
          <DialogContentText>
            Are you sure you want to delete this job posting? This action cannot be undone.
          </DialogContentText>
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setDeleteDialogOpen(false)}>Cancel</Button>
          <Button onClick={confirmDelete} color="error" variant="contained">
            Delete
          </Button>
        </DialogActions>
      </Dialog>
    </Box>
  );
};