import React, { useState, useEffect } from 'react';
import {
  Box,
  Typography,
  Button,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Grid,
  TextField,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Fab,
  Tooltip,
  useTheme,
  useMediaQuery,
  IconButton,
  Avatar,
  Chip,
  SpeedDial,
  SpeedDialAction,
  SpeedDialIcon,
} from '@mui/material';
import {
  Add,
  Edit,
  Delete,
  Visibility,
  Search,
  FilterList,
  GetApp,
  Email,
  Phone,
} from '@mui/icons-material';
import { Employee } from '../types';
import { employeeService } from '../services/employeeService';
import { ResponsiveTable, MobileCard } from '../components/Common/MobileTable';
import { MobileFilters } from '../components/Common/MobileFilters';

interface EmployeeDialogProps {
  open: boolean;
  employee: Employee | null;
  onClose: () => void;
  onSave: (employee: Partial<Employee>) => void;
  mode: 'view' | 'edit' | 'create';
}

const EmployeeDialog: React.FC<EmployeeDialogProps> = ({
  open,
  employee,
  onClose,
  onSave,
  mode,
}) => {
  const [formData, setFormData] = useState<Partial<Employee>>({
    firstName: '',
    lastName: '',
    email: '',
    phone: '',
    department: '',
    position: '',
    salary: 0,
    hireDate: '',
    status: 'active',
    address: '',
    dateOfBirth: '',
    emergencyContact: {
      name: '',
      phone: '',
      relationship: '',
    },
  });
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  useEffect(() => {
    if (employee) {
      setFormData(employee);
    } else if (mode === 'create') {
      setFormData({
        firstName: '',
        lastName: '',
        email: '',
        phone: '',
        department: '',
        position: '',
        salary: 0,
        hireDate: new Date().toISOString().split('T')[0],
        status: 'active',
        address: '',
        dateOfBirth: '',
        emergencyContact: {
          name: '',
          phone: '',
          relationship: '',
        },
      });
    }
  }, [employee, mode]);

  const handleSubmit = () => {
    onSave(formData);
  };

  const isReadonly = mode === 'view';

  return (
    <Dialog 
      open={open} 
      onClose={onClose} 
      maxWidth="md" 
      fullWidth
      fullScreen={isMobile}
      PaperProps={{
        sx: isMobile ? { 
          margin: 0,
          maxHeight: '100%',
          borderRadius: 0,
        } : undefined
      }}
    >
      <DialogTitle sx={{ 
        pb: 1,
        fontSize: { xs: '1.125rem', sm: '1.25rem' },
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'space-between'
      }}>
        <span>
          {mode === 'create' ? 'Add New Employee' : 
           mode === 'edit' ? 'Edit Employee' : 'Employee Details'}
        </span>
        {isMobile && (
          <IconButton onClick={onClose} size="small">
            <Delete />
          </IconButton>
        )}
      </DialogTitle>
      <DialogContent sx={{ px: { xs: 2, sm: 3 } }}>
        <Grid container spacing={2} sx={{ mt: 1 }}>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="First Name"
              value={formData.firstName || ''}
              onChange={(e) => setFormData({ ...formData, firstName: e.target.value })}
              disabled={isReadonly}
              required
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Last Name"
              value={formData.lastName || ''}
              onChange={(e) => setFormData({ ...formData, lastName: e.target.value })}
              disabled={isReadonly}
              required
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Email"
              type="email"
              value={formData.email || ''}
              onChange={(e) => setFormData({ ...formData, email: e.target.value })}
              disabled={isReadonly}
              required
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Phone"
              value={formData.phone || ''}
              onChange={(e) => setFormData({ ...formData, phone: e.target.value })}
              disabled={isReadonly}
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Department"
              value={formData.department || ''}
              onChange={(e) => setFormData({ ...formData, department: e.target.value })}
              disabled={isReadonly}
              required
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Position"
              value={formData.position || ''}
              onChange={(e) => setFormData({ ...formData, position: e.target.value })}
              disabled={isReadonly}
              required
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Salary"
              type="number"
              value={formData.salary || ''}
              onChange={(e) => setFormData({ ...formData, salary: Number(e.target.value) })}
              disabled={isReadonly}
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <FormControl fullWidth size={isMobile ? "medium" : "small"}>
              <InputLabel>Status</InputLabel>
              <Select
                value={formData.status || 'active'}
                onChange={(e) => setFormData({ ...formData, status: e.target.value as any })}
                disabled={isReadonly}
                label="Status"
              >
                <MenuItem value="active">Active</MenuItem>
                <MenuItem value="inactive">Inactive</MenuItem>
                <MenuItem value="terminated">Terminated</MenuItem>
              </Select>
            </FormControl>
          </Grid>
        </Grid>
      </DialogContent>
      <DialogActions sx={{ p: { xs: 2, sm: 3 }, gap: 1 }}>
        <Button onClick={onClose} fullWidth={isMobile}>
          Cancel
        </Button>
        {!isReadonly && (
          <Button 
            onClick={handleSubmit} 
            variant="contained"
            fullWidth={isMobile}
          >
            {mode === 'create' ? 'Add Employee' : 'Save Changes'}
          </Button>
        )}
      </DialogActions>
    </Dialog>
  );
};

export const MobileEmployees: React.FC = () => {
  const [employees, setEmployees] = useState<Employee[]>([]);
  const [filteredEmployees, setFilteredEmployees] = useState<Employee[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [departmentFilter, setDepartmentFilter] = useState('');
  const [statusFilter, setStatusFilter] = useState('');
  const [selectedEmployee, setSelectedEmployee] = useState<Employee | null>(null);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [dialogMode, setDialogMode] = useState<'view' | 'edit' | 'create'>('view');
  
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  useEffect(() => {
    loadEmployees();
  }, []);

  useEffect(() => {
    filterEmployees();
  }, [employees, searchTerm, departmentFilter, statusFilter]);

  const loadEmployees = async () => {
    try {
      setLoading(true);
      const data = await employeeService.getAllEmployees();
      setEmployees(data);
    } catch (error) {
      console.error('Failed to load employees:', error);
    } finally {
      setLoading(false);
    }
  };

  const filterEmployees = () => {
    let filtered = employees;

    if (searchTerm) {
      filtered = filtered.filter(emp =>
        emp.firstName.toLowerCase().includes(searchTerm.toLowerCase()) ||
        emp.lastName.toLowerCase().includes(searchTerm.toLowerCase()) ||
        emp.email.toLowerCase().includes(searchTerm.toLowerCase()) ||
        emp.employeeId.toLowerCase().includes(searchTerm.toLowerCase()) ||
        emp.department.toLowerCase().includes(searchTerm.toLowerCase()) ||
        emp.position.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    if (departmentFilter) {
      filtered = filtered.filter(emp => emp.department === departmentFilter);
    }

    if (statusFilter) {
      filtered = filtered.filter(emp => emp.status === statusFilter);
    }

    setFilteredEmployees(filtered);
  };

  const handleView = (employee: Employee) => {
    setSelectedEmployee(employee);
    setDialogMode('view');
    setDialogOpen(true);
  };

  const handleEdit = (employee: Employee) => {
    setSelectedEmployee(employee);
    setDialogMode('edit');
    setDialogOpen(true);
  };

  const handleCreate = () => {
    setSelectedEmployee(null);
    setDialogMode('create');
    setDialogOpen(true);
  };

  const handleDelete = async (employee: Employee) => {
    if (window.confirm(`Are you sure you want to delete ${employee.firstName} ${employee.lastName}?`)) {
      try {
        await employeeService.deleteEmployee(employee.id);
        loadEmployees();
      } catch (error) {
        console.error('Failed to delete employee:', error);
      }
    }
  };

  const handleSave = async (employeeData: Partial<Employee>) => {
    try {
      if (dialogMode === 'create') {
        await employeeService.createEmployee(employeeData as Omit<Employee, 'id'>);
      } else if (dialogMode === 'edit' && selectedEmployee) {
        await employeeService.updateEmployee(selectedEmployee.id, employeeData);
      }
      setDialogOpen(false);
      loadEmployees();
    } catch (error) {
      console.error('Failed to save employee:', error);
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active': return 'success';
      case 'inactive': return 'warning';
      case 'terminated': return 'error';
      default: return 'default';
    }
  };

  const departments = [...new Set(employees.map(emp => emp.department))];
  const statuses = ['active', 'inactive', 'terminated'];

  const filters = [
    {
      key: 'department',
      label: 'Department',
      type: 'select' as const,
      value: departmentFilter,
      options: departments.map(dept => ({ value: dept, label: dept }))
    },
    {
      key: 'status',
      label: 'Status',
      type: 'select' as const,
      value: statusFilter,
      options: statuses.map(status => ({ value: status, label: status.charAt(0).toUpperCase() + status.slice(1) }))
    }
  ];

  const handleFilterChange = (key: string, value: string) => {
    if (key === 'department') setDepartmentFilter(value);
    if (key === 'status') setStatusFilter(value);
  };

  const handleClearFilters = () => {
    setSearchTerm('');
    setDepartmentFilter('');
    setStatusFilter('');
  };

  // Table columns for desktop view
  const columns = [
    {
      field: 'employee',
      label: 'Employee',
      render: (value: any, row: Employee) => (
        <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
          <Avatar sx={{ width: 32, height: 32 }}>
            {row.firstName[0]}{row.lastName[0]}
          </Avatar>
          <Box>
            <Typography variant="body2" fontWeight="bold">
              {row.firstName} {row.lastName}
            </Typography>
            <Typography variant="body2" color="textSecondary">
              {row.email}
            </Typography>
          </Box>
        </Box>
      ),
    },
    { field: 'employeeId', label: 'ID' },
    { field: 'department', label: 'Department' },
    { field: 'position', label: 'Position' },
    {
      field: 'status',
      label: 'Status',
      render: (value: string) => (
        <Chip
          label={value}
          color={getStatusColor(value) as any}
          size="small"
        />
      ),
    },
    {
      field: 'actions',
      label: 'Actions',
      render: (value: any, row: Employee) => (
        <Box>
          <IconButton onClick={() => handleView(row)} size="small">
            <Visibility />
          </IconButton>
          <IconButton onClick={() => handleEdit(row)} size="small">
            <Edit />
          </IconButton>
          <IconButton onClick={() => handleDelete(row)} size="small" color="error">
            <Delete />
          </IconButton>
        </Box>
      ),
    },
  ];

  // Mobile card configuration
  const mobileCardConfig = {
    title: (row: Employee) => `${row.firstName} ${row.lastName}`,
    subtitle: (row: Employee) => `${row.employeeId} • ${row.email}`,
    avatar: (row: Employee) => `${row.firstName[0]}${row.lastName[0]}`,
    status: (row: Employee) => ({
      label: row.status,
      color: getStatusColor(row.status) as any,
    }),
    mainFields: ['department', 'position'],
    expandableFields: ['phone', 'hireDate'],
    actions: (row: Employee) => [
      <IconButton key="view" onClick={() => handleView(row)} size="small">
        <Visibility fontSize="small" />
      </IconButton>,
      <IconButton key="edit" onClick={() => handleEdit(row)} size="small">
        <Edit fontSize="small" />
      </IconButton>,
      <IconButton key="delete" onClick={() => handleDelete(row)} size="small" color="error">
        <Delete fontSize="small" />
      </IconButton>,
    ],
  };

  const speedDialActions = [
    {
      icon: <Add />,
      name: 'Add Employee',
      onClick: handleCreate,
    },
    {
      icon: <GetApp />,
      name: 'Export',
      onClick: () => console.log('Export'),
    },
    {
      icon: <Search />,
      name: 'Advanced Search',
      onClick: () => console.log('Advanced Search'),
    },
  ];

  return (
    <Box>
      <Box sx={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center', 
        mb: { xs: 2, sm: 3 },
        flexDirection: { xs: 'column', sm: 'row' },
        gap: { xs: 2, sm: 0 }
      }}>
        <Typography 
          variant={isMobile ? "h5" : "h4"} 
          sx={{ fontWeight: 600 }}
        >
          Employee Management
        </Typography>
        {!isMobile && (
          <Button
            variant="contained"
            startIcon={<Add />}
            onClick={handleCreate}
            sx={{ minWidth: 140 }}
          >
            Add Employee
          </Button>
        )}
      </Box>

      {/* Filters */}
      <MobileFilters
        searchValue={searchTerm}
        onSearchChange={setSearchTerm}
        filters={filters}
        onFilterChange={handleFilterChange}
        onClearFilters={handleClearFilters}
        placeholder="Search employees..."
      />

      {/* Employee List/Table */}
      <ResponsiveTable
        columns={columns}
        data={filteredEmployees}
        mobileCardConfig={mobileCardConfig}
        onRowClick={isMobile ? handleView : undefined}
      />

      {/* Mobile Speed Dial */}
      {isMobile && (
        <SpeedDial
          ariaLabel="Employee actions"
          sx={{ position: 'fixed', bottom: 24, right: 24 }}
          icon={<SpeedDialIcon />}
        >
          {speedDialActions.map((action) => (
            <SpeedDialAction
              key={action.name}
              icon={action.icon}
              tooltipTitle={action.name}
              onClick={action.onClick}
            />
          ))}
        </SpeedDial>
      )}

      <EmployeeDialog
        open={dialogOpen}
        employee={selectedEmployee}
        onClose={() => setDialogOpen(false)}
        onSave={handleSave}
        mode={dialogMode}
      />
    </Box>
  );
};