import React, { useState, useEffect } from 'react';
import {
  Box,
  Typography,
  Button,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Grid,
  TextField,
  IconButton,
  Avatar,
  Chip,
  SpeedDial,
  SpeedDialAction,
  SpeedDialIcon,
  useTheme,
  useMediaQuery,
  Card,
  CardContent,
} from '@mui/material';
import {
  Add,
  Edit,
  Delete,
  People,
  AttachMoney,
  Business,
  GetApp,
  Search,
} from '@mui/icons-material';
import { Department } from '../types';
import { departmentService } from '../services/departmentService';
import { MobileStatsGrid } from '../components/Common/MobileTable';

interface DepartmentDialogProps {
  open: boolean;
  department: Department | null;
  onClose: () => void;
  onSave: (department: Partial<Department>) => void;
  mode: 'view' | 'edit' | 'create';
}

const DepartmentDialog: React.FC<DepartmentDialogProps> = ({
  open,
  department,
  onClose,
  onSave,
  mode,
}) => {
  const [formData, setFormData] = useState<Partial<Department>>({
    name: '',
    description: '',
    managerId: '',
    budget: 0,
  });
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  useEffect(() => {
    if (department) {
      setFormData(department);
    } else if (mode === 'create') {
      setFormData({
        name: '',
        description: '',
        managerId: '',
        budget: 0,
      });
    }
  }, [department, mode]);

  const handleSubmit = () => {
    onSave(formData);
  };

  const isReadonly = mode === 'view';

  return (
    <Dialog 
      open={open} 
      onClose={onClose} 
      maxWidth="sm" 
      fullWidth
      fullScreen={isMobile}
      PaperProps={{
        sx: isMobile ? { 
          margin: 0,
          maxHeight: '100%',
          borderRadius: 0,
        } : undefined
      }}
    >
      <DialogTitle sx={{ 
        pb: 1,
        fontSize: { xs: '1.125rem', sm: '1.25rem' },
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'space-between'
      }}>
        <span>
          {mode === 'create' ? 'Add New Department' : 
           mode === 'edit' ? 'Edit Department' : 'Department Details'}
        </span>
        {isMobile && (
          <IconButton onClick={onClose} size="small">
            <Delete />
          </IconButton>
        )}
      </DialogTitle>
      <DialogContent sx={{ px: { xs: 2, sm: 3 } }}>
        <Grid container spacing={2} sx={{ mt: 1 }}>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Department Name"
              value={formData.name || ''}
              onChange={(e) => setFormData({ ...formData, name: e.target.value })}
              disabled={isReadonly}
              required
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Description"
              multiline
              rows={3}
              value={formData.description || ''}
              onChange={(e) => setFormData({ ...formData, description: e.target.value })}
              disabled={isReadonly}
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Manager ID"
              value={formData.managerId || ''}
              onChange={(e) => setFormData({ ...formData, managerId: e.target.value })}
              disabled={isReadonly}
              helperText="Employee ID of the department manager"
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Budget"
              type="number"
              value={formData.budget || ''}
              onChange={(e) => setFormData({ ...formData, budget: Number(e.target.value) })}
              disabled={isReadonly}
              InputProps={{
                startAdornment: '$',
              }}
              size={isMobile ? "medium" : "small"}
            />
          </Grid>
        </Grid>
      </DialogContent>
      <DialogActions sx={{ p: { xs: 2, sm: 3 }, gap: 1 }}>
        <Button onClick={onClose} fullWidth={isMobile}>
          Cancel
        </Button>
        {!isReadonly && (
          <Button 
            onClick={handleSubmit} 
            variant="contained"
            fullWidth={isMobile}
          >
            {mode === 'create' ? 'Add Department' : 'Save Changes'}
          </Button>
        )}
      </DialogActions>
    </Dialog>
  );
};

export const MobileDepartments: React.FC = () => {
  const [departments, setDepartments] = useState<Department[]>([]);
  const [loading, setLoading] = useState(true);
  const [selectedDepartment, setSelectedDepartment] = useState<Department | null>(null);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [dialogMode, setDialogMode] = useState<'view' | 'edit' | 'create'>('view');
  
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  useEffect(() => {
    loadDepartments();
  }, []);

  const loadDepartments = async () => {
    try {
      setLoading(true);
      const data = await departmentService.getAllDepartments();
      setDepartments(data);
    } catch (error) {
      console.error('Failed to load departments:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleView = (department: Department) => {
    setSelectedDepartment(department);
    setDialogMode('view');
    setDialogOpen(true);
  };

  const handleEdit = (department: Department) => {
    setSelectedDepartment(department);
    setDialogMode('edit');
    setDialogOpen(true);
  };

  const handleCreate = () => {
    setSelectedDepartment(null);
    setDialogMode('create');
    setDialogOpen(true);
  };

  const handleDelete = async (department: Department) => {
    if (window.confirm(`Are you sure you want to delete ${department.name} department?`)) {
      try {
        await departmentService.deleteDepartment(department.id);
        loadDepartments();
      } catch (error) {
        console.error('Failed to delete department:', error);
      }
    }
  };

  const handleSave = async (departmentData: Partial<Department>) => {
    try {
      if (dialogMode === 'create') {
        await departmentService.createDepartment(departmentData as Omit<Department, 'id'>);
      } else if (dialogMode === 'edit' && selectedDepartment) {
        await departmentService.updateDepartment(selectedDepartment.id, departmentData);
      }
      setDialogOpen(false);
      loadDepartments();
    } catch (error) {
      console.error('Failed to save department:', error);
    }
  };

  const totalEmployees = departments.reduce((sum, dept) => sum + dept.employeeCount, 0);
  const totalBudget = departments.reduce((sum, dept) => sum + dept.budget, 0);

  const statsData = [
    {
      title: 'Total Departments',
      value: departments.length,
      icon: <Business sx={{ fontSize: { xs: 20, sm: 24 } }} />,
      color: '#1976d2'
    },
    {
      title: 'Total Employees',
      value: totalEmployees,
      icon: <People sx={{ fontSize: { xs: 20, sm: 24 } }} />,
      color: '#2e7d32'
    },
    {
      title: 'Total Budget',
      value: `$${(totalBudget / 1000000).toFixed(1)}M`,
      icon: <AttachMoney sx={{ fontSize: { xs: 20, sm: 24 } }} />,
      color: '#ed6c02'
    }
  ];

  const speedDialActions = [
    {
      icon: <Add />,
      name: 'Add Department',
      onClick: handleCreate,
    },
    {
      icon: <GetApp />,
      name: 'Export',
      onClick: () => console.log('Export'),
    },
    {
      icon: <Search />,
      name: 'Advanced Search',
      onClick: () => console.log('Advanced Search'),
    },
  ];

  return (
    <Box>
      <Box sx={{ 
        display: 'flex', 
        justifyContent: 'space-between', 
        alignItems: 'center', 
        mb: { xs: 2, sm: 3 },
        flexDirection: { xs: 'column', sm: 'row' },
        gap: { xs: 2, sm: 0 }
      }}>
        <Typography 
          variant={isMobile ? "h5" : "h4"} 
          sx={{ fontWeight: 600 }}
        >
          Department Management
        </Typography>
        {!isMobile && (
          <Button
            variant="contained"
            startIcon={<Add />}
            onClick={handleCreate}
            sx={{ minWidth: 140 }}
          >
            Add Department
          </Button>
        )}
      </Box>
      {/* Summary Stats */}
      <MobileStatsGrid 
        stats={statsData} 
        columns={{ xs: 1, sm: 3, md: 3 }}
      />
      {/* Department Cards */}
      <Grid container spacing={2}>
        {departments.map((department) => (
          <Grid
            key={department.id}
            size={{
              xs: 12,
              sm: 6,
              md: 4
            }}>
            <Card 
              sx={{ 
                height: '100%', 
                cursor: 'pointer',
                '&:hover': {
                  boxShadow: theme => theme.shadows[4],
                }
              }}
              onClick={() => isMobile ? handleView(department) : undefined}
            >
              <CardContent sx={{ p: { xs: 2, sm: 3 } }}>
                <Box sx={{ 
                  display: 'flex', 
                  justifyContent: 'space-between', 
                  alignItems: 'flex-start', 
                  mb: 2 
                }}>
                  <Box sx={{ flex: 1 }}>
                    <Typography 
                      variant="h6" 
                      component="h3"
                      sx={{ fontSize: { xs: '1rem', sm: '1.125rem' } }}
                    >
                      {department.name}
                    </Typography>
                    <Typography 
                      variant="body2" 
                      color="textSecondary" 
                      sx={{ 
                        mb: 2, 
                        minHeight: { xs: 'auto', sm: 40 },
                        fontSize: { xs: '0.875rem', sm: '0.875rem' }
                      }}
                    >
                      {department.description}
                    </Typography>
                  </Box>
                  {!isMobile && (
                    <Box>
                      <IconButton size="small" onClick={() => handleEdit(department)}>
                        <Edit fontSize="small" />
                      </IconButton>
                      <IconButton size="small" color="error" onClick={() => handleDelete(department)}>
                        <Delete fontSize="small" />
                      </IconButton>
                    </Box>
                  )}
                </Box>

                <Box sx={{ mb: 2 }}>
                  <Box sx={{ 
                    display: 'flex', 
                    justifyContent: 'space-between', 
                    alignItems: 'center', 
                    mb: 1 
                  }}>
                    <Typography variant="body2" color="textSecondary">
                      Employees
                    </Typography>
                    <Chip
                      label={department.employeeCount}
                      size="small"
                      color="primary"
                      variant="outlined"
                    />
                  </Box>
                  <Box sx={{ 
                    display: 'flex', 
                    justifyContent: 'space-between', 
                    alignItems: 'center', 
                    mb: 1 
                  }}>
                    <Typography variant="body2" color="textSecondary">
                      Budget
                    </Typography>
                    <Typography variant="body2" fontWeight="bold">
                      ${(department.budget / 1000).toLocaleString()}K
                    </Typography>
                  </Box>
                  <Box sx={{ 
                    display: 'flex', 
                    justifyContent: 'space-between', 
                    alignItems: 'center' 
                  }}>
                    <Typography variant="body2" color="textSecondary">
                      Manager ID
                    </Typography>
                    <Typography variant="body2">
                      {department.managerId || 'Not assigned'}
                    </Typography>
                  </Box>
                </Box>

                <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                  <People fontSize="small" color="action" />
                  <Typography 
                    variant="body2" 
                    color="textSecondary"
                    sx={{ fontSize: { xs: '0.75rem', sm: '0.875rem' } }}
                  >
                    {department.employeeCount > 0 
                      ? `${Math.round(department.budget / department.employeeCount / 1000)}K avg per employee`
                      : 'No employees assigned'
                    }
                  </Typography>
                </Box>

                {/* Mobile action buttons */}
                {isMobile && (
                  <Box sx={{ 
                    display: 'flex', 
                    justifyContent: 'flex-end', 
                    gap: 1, 
                    mt: 2,
                    pt: 2,
                    borderTop: '1px solid',
                    borderColor: 'divider'
                  }}>
                    <IconButton 
                      size="small" 
                      onClick={(e) => {
                        e.stopPropagation();
                        handleEdit(department);
                      }}
                    >
                      <Edit fontSize="small" />
                    </IconButton>
                    <IconButton 
                      size="small" 
                      color="error" 
                      onClick={(e) => {
                        e.stopPropagation();
                        handleDelete(department);
                      }}
                    >
                      <Delete fontSize="small" />
                    </IconButton>
                  </Box>
                )}
              </CardContent>
            </Card>
          </Grid>
        ))}
      </Grid>
      {/* Mobile Speed Dial */}
      {isMobile && (
        <SpeedDial
          ariaLabel="Department actions"
          sx={{ position: 'fixed', bottom: 24, right: 24 }}
          icon={<SpeedDialIcon />}
        >
          {speedDialActions.map((action) => (
            <SpeedDialAction
              key={action.name}
              icon={action.icon}
              tooltipTitle={action.name}
              onClick={action.onClick}
            />
          ))}
        </SpeedDial>
      )}
      <DepartmentDialog
        open={dialogOpen}
        department={selectedDepartment}
        onClose={() => setDialogOpen(false)}
        onSave={handleSave}
        mode={dialogMode}
      />
    </Box>
  );
};