import React, { useState, useEffect } from 'react';
import {
  Box,
  Typography,
  Paper,
  TextField,
  Button,
  Tabs,
  Tab,
  Card,
  CardContent,
  CardActions,
  Alert,
  Snackbar,
  CircularProgress,
  Chip,
  IconButton,
  List,
  ListItem,
  ListItemText,
  ListItemSecondaryAction,
  Divider,
  MenuItem,
  Select,
  FormControl,
  InputLabel,
  LinearProgress,
} from '@mui/material';
import {
  Email,
  Settings,
  Add,
  Edit,
  Send,
  Inbox,
  Science,
  Visibility,
} from '@mui/icons-material';
import { 
  MailSettings as IMailSettings, 
  SMTPConfig, 
  IMAPConfig, 
  EmailTemplate,
  SentEmail,
  ReceivedEmail,
} from '../types/services/mailService.types';
import { mailService } from '../services';
import { SettingsDialog } from '../components/Mail/SettingsDialog';

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

function TabPanel(props: TabPanelProps) {
  const { children, value, index, ...other } = props;

  return (
    <div
      role="tabpanel"
      hidden={value !== index}
      id={`mail-tabpanel-${index}`}
      aria-labelledby={`mail-tab-${index}`}
      {...other}
    >
      {value === index && <Box sx={{ p: 3 }}>{children}</Box>}
    </div>
  );
}

export const MailSettings: React.FC = () => {
  const [tabValue, setTabValue] = useState(0);
  const [settings, setSettings] = useState<IMailSettings[]>([]);
  const [templates, setTemplates] = useState<EmailTemplate[]>([]);
  const [sentEmails, setSentEmails] = useState<SentEmail[]>([]);
  const [receivedEmails, setReceivedEmails] = useState<ReceivedEmail[]>([]);
  const [loading, setLoading] = useState(false);
  const [snackbar, setSnackbar] = useState({ open: false, message: '', severity: 'success' as 'success' | 'error' });
  
  // Settings Dialog State
  const [settingsDialog, setSettingsDialog] = useState(false);
  const [editingSettings, setEditingSettings] = useState<IMailSettings | null>(null);
  const [testingConnection, setTestingConnection] = useState(false);
  const [emailForm, setEmailForm] = useState({
    to: '',
    subject: '',
    content: '',
    templateId: '',
  });

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    setLoading(true);
    try {
      const [settingsRes, templatesRes, sentRes, receivedRes] = await Promise.all([
        mailService.getMailSettings(),
        mailService.getEmailTemplates(),
        mailService.getSentEmails(),
        mailService.getReceivedEmails(),
      ]);

      if (settingsRes.success) setSettings(settingsRes.data || []);
      if (templatesRes.success) setTemplates(templatesRes.data || []);
      if (sentRes.success) setSentEmails(sentRes.data?.emails || []);
      if (receivedRes.success) setReceivedEmails(receivedRes.data?.emails || []);
    } catch (error) {
      showSnackbar('Failed to load mail data', 'error');
    } finally {
      setLoading(false);
    }
  };

  const showSnackbar = (message: string, severity: 'success' | 'error') => {
    setSnackbar({ open: true, message, severity });
  };

  const handleSaveSettings = async (formData: any) => {
    setLoading(true);
    try {
      const settingsData: Omit<IMailSettings, 'id' | 'createdAt' | 'updatedAt'> = {
        name: formData.name,
        smtp: {
          host: formData.smtpHost,
          port: parseInt(formData.smtpPort),
          secure: formData.smtpSecure,
          auth: {
            user: formData.smtpUser,
            pass: formData.smtpPass,
          },
        },
        imap: {
          host: formData.imapHost,
          port: parseInt(formData.imapPort),
          secure: formData.imapSecure,
          auth: {
            user: formData.imapUser,
            pass: formData.imapPass,
          },
        },
        defaultFrom: formData.defaultFrom,
        defaultFromName: formData.defaultFromName,
        isDefault: formData.isDefault,
        isActive: formData.isActive,
      };

      const result = await mailService.saveMailSettings(settingsData);
      if (result.success) {
        showSnackbar('Settings saved successfully', 'success');
        loadData();
        setSettingsDialog(false);
        setEditingSettings(null);
      } else {
        showSnackbar(result.error || 'Failed to save settings', 'error');
      }
    } catch (error) {
      showSnackbar('Failed to save settings', 'error');
    } finally {
      setLoading(false);
    }
  };

  const handleTestConnection = async (smtp: SMTPConfig, imap: IMAPConfig) => {
    setTestingConnection(true);
    try {
      const result = await mailService.testConnection({ smtp, imap });
      if (result.success && result.data) {
        if (result.data.smtpStatus && result.data.imapStatus) {
          showSnackbar('Connection test successful', 'success');
        } else {
          showSnackbar('Some connections failed', 'error');
        }
      }
    } catch (error) {
      showSnackbar('Connection test failed', 'error');
    } finally {
      setTestingConnection(false);
    }
  };

  const handleSendTestEmail = async () => {
    if (!emailForm.to || !emailForm.subject) {
      showSnackbar('Please fill in required fields', 'error');
      return;
    }

    setLoading(true);
    try {
      const message = {
        to: emailForm.to,
        subject: emailForm.subject,
        html: `<p>${emailForm.content}</p>`,
        text: emailForm.content,
        templateId: emailForm.templateId || undefined,
      };

      const result = await mailService.sendEmail(message);
      if (result.success) {
        showSnackbar('Email sent successfully', 'success');
        setEmailForm({ to: '', subject: '', content: '', templateId: '' });
        loadData();
      } else {
        showSnackbar(result.error || 'Failed to send email', 'error');
      }
    } catch (error) {
      showSnackbar('Failed to send email', 'error');
    } finally {
      setLoading(false);
    }
  };

  const SettingsTab = () => (
    <Box>
      <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 3 }}>
        <Typography variant="h6">Mail Settings</Typography>
        <Button
          variant="contained"
          startIcon={<Add />}
          onClick={() => {
            setEditingSettings(null);
            setSettingsDialog(true);
          }}
        >
          Add Settings
        </Button>
      </Box>

      {settings.map((setting) => (
        <Card key={setting.id} sx={{ mb: 2 }}>
          <CardContent>
            <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'flex-start' }}>
              <Box>
                <Typography variant="h6">{setting.name}</Typography>
                <Typography variant="body2" color="text.secondary">
                  SMTP: {setting.smtp.host}:{setting.smtp.port}
                </Typography>
                <Typography variant="body2" color="text.secondary">
                  IMAP: {setting.imap.host}:{setting.imap.port}
                </Typography>
                <Typography variant="body2" color="text.secondary">
                  From: {setting.defaultFromName} &lt;{setting.defaultFrom}&gt;
                </Typography>
                <Box sx={{ mt: 1 }}>
                  {setting.isDefault && <Chip label="Default" size="small" color="primary" sx={{ mr: 1 }} />}
                  <Chip 
                    label={setting.isActive ? "Active" : "Inactive"} 
                    size="small" 
                    color={setting.isActive ? "success" : "default"} 
                  />
                </Box>
              </Box>
              <Box>
                <IconButton
                  onClick={() => {
                    setEditingSettings(setting);
                    setSettingsDialog(true);
                  }}
                >
                  <Edit />
                </IconButton>
                <IconButton
                  onClick={() => handleTestConnection(setting.smtp, setting.imap)}
                  disabled={testingConnection}
                >
                  <Science />
                </IconButton>
              </Box>
            </Box>
          </CardContent>
        </Card>
      ))}
    </Box>
  );

  const TemplatesTab = () => (
    <Box>
      <Box sx={{ display: 'flex', justifyContent: 'space-between', mb: 3 }}>
        <Typography variant="h6">Email Templates</Typography>
        <Button
          variant="contained"
          startIcon={<Add />}
          onClick={() => {}}
        >
          Add Template
        </Button>
      </Box>

      <Box sx={{ display: 'flex', flexWrap: 'wrap', gap: 2 }}>
        {templates.map((template) => (
          <Box key={template.id} sx={{ flex: '1 1 300px', minWidth: '300px' }}>
            <Card>
              <CardContent>
                <Typography variant="h6" gutterBottom>
                  {template.name}
                </Typography>
                <Typography variant="body2" color="text.secondary" gutterBottom>
                  Category: {template.category.replace('_', ' ')}
                </Typography>
                <Typography variant="body2" gutterBottom>
                  Subject: {template.subject}
                </Typography>
                <Box sx={{ mt: 1 }}>
                  <Chip 
                    label={template.isActive ? "Active" : "Inactive"} 
                    size="small" 
                    color={template.isActive ? "success" : "default"} 
                  />
                </Box>
              </CardContent>
              <CardActions>
                <Button size="small" startIcon={<Edit />}>
                  Edit
                </Button>
                <Button size="small" startIcon={<Visibility />}>
                  Preview
                </Button>
              </CardActions>
            </Card>
          </Box>
        ))}
      </Box>
    </Box>
  );

  const SendEmailTab = () => (
    <Box>
      <Typography variant="h6" gutterBottom>Send Email</Typography>
      <Paper sx={{ p: 3 }}>
        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
          <TextField
            label="To"
            value={emailForm.to}
            onChange={(e) => setEmailForm({ ...emailForm, to: e.target.value })}
            fullWidth
            required
          />
          <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
            <Box sx={{ flex: '2 1 300px', minWidth: '250px' }}>
              <TextField
                label="Subject"
                value={emailForm.subject}
                onChange={(e) => setEmailForm({ ...emailForm, subject: e.target.value })}
                fullWidth
                required
              />
            </Box>
            <Box sx={{ flex: '1 1 200px', minWidth: '150px' }}>
              <FormControl fullWidth>
                <InputLabel>Template</InputLabel>
                <Select
                  value={emailForm.templateId}
                  onChange={(e) => setEmailForm({ ...emailForm, templateId: e.target.value })}
                >
                  <MenuItem value="">None</MenuItem>
                  {templates.map((template) => (
                    <MenuItem key={template.id} value={template.id}>
                      {template.name}
                    </MenuItem>
                  ))}
                </Select>
              </FormControl>
            </Box>
          </Box>
          <TextField
            label="Content"
            value={emailForm.content}
            onChange={(e) => setEmailForm({ ...emailForm, content: e.target.value })}
            multiline
            rows={8}
            fullWidth
          />
          <Button
            variant="contained"
            startIcon={<Send />}
            onClick={handleSendTestEmail}
            disabled={loading}
            sx={{ alignSelf: 'flex-start' }}
          >
            {loading ? <CircularProgress size={20} /> : 'Send Email'}
          </Button>
        </Box>
      </Paper>
    </Box>
  );

  const SentEmailsTab = () => (
    <Box>
      <Typography variant="h6" gutterBottom>Sent Emails</Typography>
      <List>
        {sentEmails.map((email) => (
          <React.Fragment key={email.id}>
            <ListItem>
              <ListItemText
                primary={email.subject}
                secondary={`To: ${email.to.join(', ')} • ${email.sentAt.toLocaleString()}`}
              />
              <ListItemSecondaryAction>
                <Chip 
                  label={email.status} 
                  size="small" 
                  color={email.status === 'sent' ? 'success' : 'default'} 
                />
              </ListItemSecondaryAction>
            </ListItem>
            <Divider />
          </React.Fragment>
        ))}
      </List>
    </Box>
  );

  const ReceivedEmailsTab = () => (
    <Box>
      <Typography variant="h6" gutterBottom>Received Emails</Typography>
      <List>
        {receivedEmails.map((email) => (
          <React.Fragment key={email.id}>
            <ListItem>
              <ListItemText
                primary={email.subject}
                secondary={`From: ${email.from} • ${email.receivedAt.toLocaleString()}`}
              />
              <ListItemSecondaryAction>
                <Chip 
                  label={email.isRead ? "Read" : "Unread"} 
                  size="small" 
                  color={email.isRead ? 'default' : 'primary'} 
                />
              </ListItemSecondaryAction>
            </ListItem>
            <Divider />
          </React.Fragment>
        ))}
      </List>
    </Box>
  );

  return (
    <Box sx={{ p: 3 }}>
      <Typography variant="h4" gutterBottom>
        Mail Settings
      </Typography>

      {loading && <LinearProgress sx={{ mb: 2 }} />}

      <Paper sx={{ width: '100%' }}>
        <Tabs
          value={tabValue}
          onChange={(_, newValue) => setTabValue(newValue)}
          indicatorColor="primary"
          textColor="primary"
        >
          <Tab icon={<Settings />} label="Settings" />
          <Tab icon={<Email />} label="Templates" />
          <Tab icon={<Send />} label="Send Email" />
          <Tab icon={<Inbox />} label="Sent" />
          <Tab icon={<Email />} label="Received" />
        </Tabs>

        <TabPanel value={tabValue} index={0}>
          <SettingsTab />
        </TabPanel>
        <TabPanel value={tabValue} index={1}>
          <TemplatesTab />
        </TabPanel>
        <TabPanel value={tabValue} index={2}>
          <SendEmailTab />
        </TabPanel>
        <TabPanel value={tabValue} index={3}>
          <SentEmailsTab />
        </TabPanel>
        <TabPanel value={tabValue} index={4}>
          <ReceivedEmailsTab />
        </TabPanel>
      </Paper>

      <Snackbar
        open={snackbar.open}
        autoHideDuration={6000}
        onClose={() => setSnackbar({ ...snackbar, open: false })}
      >
        <Alert severity={snackbar.severity} onClose={() => setSnackbar({ ...snackbar, open: false })}>
          {snackbar.message}
        </Alert>
      </Snackbar>

      <SettingsDialog
        open={settingsDialog}
        onClose={() => {
          setSettingsDialog(false);
          setEditingSettings(null);
        }}
        onSave={handleSaveSettings}
        editingSettings={editingSettings}
        loading={loading}
        onTestConnection={handleTestConnection}
        testingConnection={testingConnection}
      />
    </Box>
  );
};