import React, { useState, useEffect } from 'react';
import {
  Box,
  Paper,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TablePagination,
  TableRow,
  Typography,
  Button,
  TextField,
  InputAdornment,
  Chip,
  Card,
  CardContent,
  Grid,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Avatar,
  IconButton,
  Tabs,
  Tab,
  Badge,
  Divider,
} from '@mui/material';
import {
  Search,
  Add,
  Visibility,
  CheckCircle,
  Cancel,
  Pending,
  EventNote,
  CalendarMonth,
  FilterList,
} from '@mui/icons-material';
import { LeaveRequest } from '../types';
import { format, differenceInDays } from 'date-fns';
import { useFormValidation, VALIDATION_SCHEMAS } from '../utils/validation';

const mockLeaves: LeaveRequest[] = [
  {
    id: '1',
    employeeId: '1',
    type: 'vacation',
    startDate: '2023-06-25',
    endDate: '2023-06-30',
    days: 6,
    reason: 'Summer vacation with family',
    status: 'approved',
    approvedBy: 'HR001',
    appliedDate: '2023-06-15',
    comments: 'Approved for summer break'
  },
  {
    id: '2',
    employeeId: '2',
    type: 'sick',
    startDate: '2023-06-20',
    endDate: '2023-06-21',
    days: 2,
    reason: 'Medical appointment and recovery',
    status: 'pending',
    appliedDate: '2023-06-19',
  },
  {
    id: '3',
    employeeId: '3',
    type: 'personal',
    startDate: '2023-07-01',
    endDate: '2023-07-01',
    days: 1,
    reason: 'Personal matters',
    status: 'rejected',
    approvedBy: 'HR001',
    appliedDate: '2023-06-18',
    comments: 'Conflicting project deadline'
  },
  {
    id: '4',
    employeeId: '4',
    type: 'maternity',
    startDate: '2023-07-15',
    endDate: '2023-10-15',
    days: 90,
    reason: 'Maternity leave',
    status: 'approved',
    approvedBy: 'HR001',
    appliedDate: '2023-06-10',
    comments: 'Standard maternity leave approved'
  },
  {
    id: '5',
    employeeId: '5',
    type: 'vacation',
    startDate: '2023-08-01',
    endDate: '2023-08-05',
    days: 5,
    reason: 'Annual vacation',
    status: 'pending',
    appliedDate: '2023-06-20',
  },
];

const mockEmployeeNames: { [key: string]: string } = {
  '1': 'John Doe',
  '2': 'Sarah Johnson',
  '3': 'Michael Brown',
  '4': 'Emily Davis',
  '5': 'David Wilson',
};

interface LeaveDialogProps {
  open: boolean;
  leave: LeaveRequest | null;
  onClose: () => void;
  onSave?: (leave: Partial<LeaveRequest>) => void;
  onApprove?: (leaveId: string, comments?: string) => void;
  onReject?: (leaveId: string, comments?: string) => void;
  mode: 'view' | 'edit' | 'create';
}

const LeaveDialog: React.FC<LeaveDialogProps> = ({
  open,
  leave,
  onClose,
  onSave,
  onApprove,
  onReject,
  mode,
}) => {
  const [formData, setFormData] = useState<Partial<LeaveRequest>>({
    employeeId: '',
    type: 'vacation',
    startDate: '',
    endDate: '',
    reason: '',
    status: 'pending',
  });
  const [comments, setComments] = useState('');

  const {
    validateForm,
    handleFieldChange,
    clearErrors,
    hasError,
    getErrorMessage,
  } = useFormValidation(VALIDATION_SCHEMAS.LEAVE);

  useEffect(() => {
    if (leave) {
      setFormData(leave);
      setComments(leave.comments || '');
    } else if (mode === 'create') {
      setFormData({
        employeeId: '',
        type: 'vacation',
        startDate: format(new Date(), 'yyyy-MM-dd'),
        endDate: format(new Date(), 'yyyy-MM-dd'),
        reason: '',
        status: 'pending',
      });
    }
  }, [leave, mode]);

  useEffect(() => {
    if (formData.startDate && formData.endDate) {
      const days = differenceInDays(new Date(formData.endDate), new Date(formData.startDate)) + 1;
      setFormData(prev => ({ ...prev, days: days > 0 ? days : 1 }));
    }
  }, [formData.startDate, formData.endDate]);

  const handleFieldChangeWithValidation = (fieldName: string, value: any) => {
    setFormData(prev => ({ ...prev, [fieldName]: value }));
    handleFieldChange(fieldName, value);
  };

  const handleSubmit = () => {
    const validation = validateForm(formData);
    if (!validation.isValid) {
      return;
    }
    if (onSave) {
      onSave({
        ...formData,
        appliedDate: formData.appliedDate || format(new Date(), 'yyyy-MM-dd'),
      });
      clearErrors();
    }
  };

  const handleApprove = () => {
    if (onApprove && leave) {
      onApprove(leave.id, comments);
    }
  };

  const handleReject = () => {
    if (onReject && leave) {
      onReject(leave.id, comments);
    }
  };

  const isReadonly = mode === 'view';

  return (
    <Dialog open={open} onClose={onClose} maxWidth="md" fullWidth>
      <DialogTitle>
        {mode === 'create' ? 'Apply for Leave' : 
         mode === 'edit' ? 'Edit Leave Request' : 'Leave Request Details'}
      </DialogTitle>
      <DialogContent>
        <Grid container spacing={2} sx={{ mt: 1 }}>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Employee ID"
              value={formData.employeeId || ''}
              onChange={(e) => handleFieldChangeWithValidation('employeeId', e.target.value)}
              disabled={isReadonly}
              required
              error={hasError('employeeId')}
              helperText={getErrorMessage('employeeId')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <FormControl fullWidth>
              <InputLabel>Leave Type</InputLabel>
              <Select
                value={formData.type || 'vacation'}
                onChange={(e) => handleFieldChangeWithValidation('type', e.target.value)}
                disabled={isReadonly}
                error={hasError('type')}
              >
                <MenuItem value="sick">Sick Leave</MenuItem>
                <MenuItem value="vacation">Vacation</MenuItem>
                <MenuItem value="personal">Personal Leave</MenuItem>
                <MenuItem value="maternity">Maternity Leave</MenuItem>
                <MenuItem value="paternity">Paternity Leave</MenuItem>
              </Select>
            </FormControl>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Start Date"
              type="date"
              value={formData.startDate || ''}
              onChange={(e) => handleFieldChangeWithValidation('startDate', e.target.value)}
              disabled={isReadonly}
              InputLabelProps={{ shrink: true }}
              required
              error={hasError('startDate')}
              helperText={getErrorMessage('startDate')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="End Date"
              type="date"
              value={formData.endDate || ''}
              onChange={(e) => handleFieldChangeWithValidation('endDate', e.target.value)}
              disabled={isReadonly}
              InputLabelProps={{ shrink: true }}
              required
              error={hasError('endDate')}
              helperText={getErrorMessage('endDate')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Total Days"
              value={formData.days || ''}
              disabled
              InputProps={{
                readOnly: true,
              }}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Applied Date"
              type="date"
              value={formData.appliedDate || ''}
              disabled
              InputLabelProps={{ shrink: true }}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Reason"
              multiline
              rows={3}
              value={formData.reason || ''}
              onChange={(e) => handleFieldChangeWithValidation('reason', e.target.value)}
              disabled={isReadonly}
              required
              error={hasError('reason')}
              helperText={getErrorMessage('reason')}
            />
          </Grid>
          {leave && leave.status !== 'pending' && (
            <Grid size={12}>
              <TextField
                fullWidth
                label="Status"
                value={formData.status}
                disabled
              />
            </Grid>
          )}
          {(mode === 'view' && leave?.status === 'pending') && (
            <Grid size={12}>
              <Divider sx={{ my: 2 }} />
              <Typography variant="h6" gutterBottom>
                Management Action
              </Typography>
              <TextField
                fullWidth
                label="Comments"
                multiline
                rows={2}
                value={comments}
                onChange={(e) => setComments(e.target.value)}
                placeholder="Add comments for approval/rejection"
              />
            </Grid>
          )}
          {leave?.comments && (
            <Grid size={12}>
              <TextField
                fullWidth
                label="Management Comments"
                multiline
                rows={2}
                value={leave.comments}
                disabled
              />
            </Grid>
          )}
        </Grid>
      </DialogContent>
      <DialogActions>
        <Button onClick={onClose}>Close</Button>
        {mode === 'view' && leave?.status === 'pending' && (
          <>
            <Button onClick={handleReject} color="error">
              Reject
            </Button>
            <Button onClick={handleApprove} variant="contained" color="success">
              Approve
            </Button>
          </>
        )}
        {!isReadonly && (
          <Button onClick={handleSubmit} variant="contained">
            {mode === 'create' ? 'Submit Request' : 'Save Changes'}
          </Button>
        )}
      </DialogActions>
    </Dialog>
  );
};

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

const TabPanel: React.FC<TabPanelProps> = ({ children, value, index }) => (
  <div hidden={value !== index}>
    {value === index && <Box sx={{ py: 3 }}>{children}</Box>}
  </div>
);

export const Leaves: React.FC = () => {
  const [leaves, setLeaves] = useState<LeaveRequest[]>(mockLeaves);
  const [filteredLeaves, setFilteredLeaves] = useState<LeaveRequest[]>(mockLeaves);
  const [searchTerm, setSearchTerm] = useState('');
  const [statusFilter, setStatusFilter] = useState('');
  const [typeFilter, setTypeFilter] = useState('');
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [selectedLeave, setSelectedLeave] = useState<LeaveRequest | null>(null);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [dialogMode, setDialogMode] = useState<'view' | 'edit' | 'create'>('view');
  const [tabValue, setTabValue] = useState(0);

  useEffect(() => {
    filterLeaves();
  }, [leaves, searchTerm, statusFilter, typeFilter]);

  const filterLeaves = () => {
    let filtered = leaves;

    if (searchTerm) {
      filtered = filtered.filter(leave =>
        mockEmployeeNames[leave.employeeId]?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        leave.employeeId.toLowerCase().includes(searchTerm.toLowerCase()) ||
        leave.reason.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    if (statusFilter) {
      filtered = filtered.filter(leave => leave.status === statusFilter);
    }

    if (typeFilter) {
      filtered = filtered.filter(leave => leave.type === typeFilter);
    }

    setFilteredLeaves(filtered);
    setPage(0);
  };

  const handleView = (leave: LeaveRequest) => {
    setSelectedLeave(leave);
    setDialogMode('view');
    setDialogOpen(true);
  };

  const handleCreate = () => {
    setSelectedLeave(null);
    setDialogMode('create');
    setDialogOpen(true);
  };

  const handleSave = (leaveData: Partial<LeaveRequest>) => {
    if (dialogMode === 'create') {
      const newLeave: LeaveRequest = {
        ...leaveData,
        id: Date.now().toString(),
        appliedDate: format(new Date(), 'yyyy-MM-dd'),
      } as LeaveRequest;
      setLeaves(prev => [...prev, newLeave]);
    }
    setDialogOpen(false);
  };

  const handleApprove = (leaveId: string, comments?: string) => {
    setLeaves(prev =>
      prev.map(leave =>
        leave.id === leaveId
          ? { ...leave, status: 'approved', approvedBy: 'HR001', comments }
          : leave
      )
    );
    setDialogOpen(false);
  };

  const handleReject = (leaveId: string, comments?: string) => {
    setLeaves(prev =>
      prev.map(leave =>
        leave.id === leaveId
          ? { ...leave, status: 'rejected', approvedBy: 'HR001', comments }
          : leave
      )
    );
    setDialogOpen(false);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'approved': return 'success';
      case 'rejected': return 'error';
      case 'pending': return 'warning';
      default: return 'default';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'approved': return <CheckCircle />;
      case 'rejected': return <Cancel />;
      case 'pending': return <Pending />;
      default: return <Pending />;
    }
  };

  const getLeaveTypeColor = (type: string) => {
    switch (type) {
      case 'sick': return 'error';
      case 'vacation': return 'primary';
      case 'personal': return 'info';
      case 'maternity': return 'secondary';
      case 'paternity': return 'secondary';
      default: return 'default';
    }
  };

  const leaveStats = {
    total: leaves.length,
    pending: leaves.filter(l => l.status === 'pending').length,
    approved: leaves.filter(l => l.status === 'approved').length,
    rejected: leaves.filter(l => l.status === 'rejected').length,
    totalDays: leaves.filter(l => l.status === 'approved').reduce((sum, l) => sum + l.days, 0),
  };

  return (
    <Box>
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
        <Typography variant="h4" gutterBottom>
          Leave Management
        </Typography>
        <Button variant="contained" startIcon={<Add />} onClick={handleCreate}>
          Apply Leave
        </Button>
      </Box>
      <Box sx={{ borderBottom: 1, borderColor: 'divider', mb: 3 }}>
        <Tabs value={tabValue} onChange={(e, newValue) => setTabValue(newValue)}>
          <Tab label="All Requests" />
          <Tab 
            label={
              <Badge badgeContent={leaveStats.pending} color="warning">
                Pending Approval
              </Badge>
            } 
          />
          <Tab label="Leave Analytics" />
        </Tabs>
      </Box>
      <TabPanel value={tabValue} index={0}>
        {/* Stats Cards */}
        <Grid container spacing={3} sx={{ mb: 3 }}>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 3
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <EventNote sx={{ fontSize: 40, color: 'primary.main', mb: 1 }} />
                <Typography variant="h4">{leaveStats.total}</Typography>
                <Typography color="textSecondary">Total Requests</Typography>
              </CardContent>
            </Card>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 3
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <Badge badgeContent={leaveStats.pending} color="warning">
                  <Pending sx={{ fontSize: 40, color: 'warning.main', mb: 1 }} />
                </Badge>
                <Typography variant="h4">{leaveStats.pending}</Typography>
                <Typography color="textSecondary">Pending</Typography>
              </CardContent>
            </Card>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 3
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <CheckCircle sx={{ fontSize: 40, color: 'success.main', mb: 1 }} />
                <Typography variant="h4">{leaveStats.approved}</Typography>
                <Typography color="textSecondary">Approved</Typography>
              </CardContent>
            </Card>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 3
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <CalendarMonth sx={{ fontSize: 40, color: 'info.main', mb: 1 }} />
                <Typography variant="h4">{leaveStats.totalDays}</Typography>
                <Typography color="textSecondary">Approved Days</Typography>
              </CardContent>
            </Card>
          </Grid>
        </Grid>

        {/* Filters */}
        <Paper sx={{ mb: 2, p: 2 }}>
          <Box sx={{ display: 'flex', gap: 2, alignItems: 'center', flexWrap: 'wrap' }}>
            <TextField
              size="small"
              placeholder="Search requests..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              InputProps={{
                startAdornment: (
                  <InputAdornment position="start">
                    <Search />
                  </InputAdornment>
                ),
              }}
              sx={{ minWidth: 250 }}
            />
            <FormControl size="small" sx={{ minWidth: 120 }}>
              <InputLabel>Status</InputLabel>
              <Select
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
                label="Status"
              >
                <MenuItem value="">All Status</MenuItem>
                <MenuItem value="pending">Pending</MenuItem>
                <MenuItem value="approved">Approved</MenuItem>
                <MenuItem value="rejected">Rejected</MenuItem>
              </Select>
            </FormControl>
            <FormControl size="small" sx={{ minWidth: 120 }}>
              <InputLabel>Type</InputLabel>
              <Select
                value={typeFilter}
                onChange={(e) => setTypeFilter(e.target.value)}
                label="Type"
              >
                <MenuItem value="">All Types</MenuItem>
                <MenuItem value="sick">Sick</MenuItem>
                <MenuItem value="vacation">Vacation</MenuItem>
                <MenuItem value="personal">Personal</MenuItem>
                <MenuItem value="maternity">Maternity</MenuItem>
                <MenuItem value="paternity">Paternity</MenuItem>
              </Select>
            </FormControl>
            <Button
              startIcon={<FilterList />}
              onClick={() => {
                setSearchTerm('');
                setStatusFilter('');
                setTypeFilter('');
              }}
            >
              Clear
            </Button>
          </Box>
        </Paper>

        {/* Leave Requests Table */}
        <TableContainer component={Paper}>
          <Table>
            <TableHead>
              <TableRow>
                <TableCell>Employee</TableCell>
                <TableCell>Type</TableCell>
                <TableCell>Duration</TableCell>
                <TableCell>Days</TableCell>
                <TableCell>Applied Date</TableCell>
                <TableCell>Status</TableCell>
                <TableCell>Actions</TableCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {filteredLeaves
                .slice(page * rowsPerPage, page * rowsPerPage + rowsPerPage)
                .map((leave) => (
                  <TableRow key={leave.id} hover>
                    <TableCell>
                      <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                        <Avatar sx={{ width: 32, height: 32 }}>
                          {mockEmployeeNames[leave.employeeId]?.split(' ').map(n => n[0]).join('')}
                        </Avatar>
                        <Box>
                          <Typography variant="body2" fontWeight="bold">
                            {mockEmployeeNames[leave.employeeId] || `Employee ${leave.employeeId}`}
                          </Typography>
                          <Typography variant="body2" color="textSecondary">
                            ID: {leave.employeeId}
                          </Typography>
                        </Box>
                      </Box>
                    </TableCell>
                    <TableCell>
                      <Chip
                        label={leave.type}
                        color={getLeaveTypeColor(leave.type) as any}
                        size="small"
                        variant="outlined"
                      />
                    </TableCell>
                    <TableCell>
                      <Typography variant="body2">
                        {format(new Date(leave.startDate), 'MMM dd')} - {format(new Date(leave.endDate), 'MMM dd, yyyy')}
                      </Typography>
                    </TableCell>
                    <TableCell>
                      <Typography variant="body2" fontWeight="bold">
                        {leave.days} days
                      </Typography>
                    </TableCell>
                    <TableCell>
                      {format(new Date(leave.appliedDate), 'MMM dd, yyyy')}
                    </TableCell>
                    <TableCell>
                      <Chip
                        icon={getStatusIcon(leave.status)}
                        label={leave.status}
                        color={getStatusColor(leave.status) as any}
                        size="small"
                      />
                    </TableCell>
                    <TableCell>
                      <IconButton onClick={() => handleView(leave)} size="small">
                        <Visibility />
                      </IconButton>
                    </TableCell>
                  </TableRow>
                ))}
            </TableBody>
          </Table>
          <TablePagination
            rowsPerPageOptions={[5, 10, 25]}
            component="div"
            count={filteredLeaves.length}
            rowsPerPage={rowsPerPage}
            page={page}
            onPageChange={(event, newPage) => setPage(newPage)}
            onRowsPerPageChange={(event) => {
              setRowsPerPage(parseInt(event.target.value, 10));
              setPage(0);
            }}
          />
        </TableContainer>
      </TabPanel>
      <TabPanel value={tabValue} index={1}>
        {/* Pending requests only */}
        <Typography variant="h6" gutterBottom>
          Requests Awaiting Approval ({leaveStats.pending})
        </Typography>
        <TableContainer component={Paper}>
          <Table>
            <TableHead>
              <TableRow>
                <TableCell>Employee</TableCell>
                <TableCell>Type</TableCell>
                <TableCell>Duration</TableCell>
                <TableCell>Days</TableCell>
                <TableCell>Reason</TableCell>
                <TableCell>Actions</TableCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {leaves.filter(leave => leave.status === 'pending').map((leave) => (
                <TableRow key={leave.id} hover>
                  <TableCell>
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                      <Avatar sx={{ width: 32, height: 32 }}>
                        {mockEmployeeNames[leave.employeeId]?.split(' ').map(n => n[0]).join('')}
                      </Avatar>
                      <Box>
                        <Typography variant="body2" fontWeight="bold">
                          {mockEmployeeNames[leave.employeeId]}
                        </Typography>
                        <Typography variant="body2" color="textSecondary">
                          ID: {leave.employeeId}
                        </Typography>
                      </Box>
                    </Box>
                  </TableCell>
                  <TableCell>
                    <Chip
                      label={leave.type}
                      color={getLeaveTypeColor(leave.type) as any}
                      size="small"
                      variant="outlined"
                    />
                  </TableCell>
                  <TableCell>
                    <Typography variant="body2">
                      {format(new Date(leave.startDate), 'MMM dd')} - {format(new Date(leave.endDate), 'MMM dd, yyyy')}
                    </Typography>
                  </TableCell>
                  <TableCell>
                    <Typography variant="body2" fontWeight="bold">
                      {leave.days} days
                    </Typography>
                  </TableCell>
                  <TableCell>
                    <Typography variant="body2" sx={{ maxWidth: 200 }}>
                      {leave.reason.length > 50 ? leave.reason.substring(0, 50) + '...' : leave.reason}
                    </Typography>
                  </TableCell>
                  <TableCell>
                    <IconButton onClick={() => handleView(leave)} size="small">
                      <Visibility />
                    </IconButton>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </TableContainer>
      </TabPanel>
      <TabPanel value={tabValue} index={2}>
        <Typography variant="h6" gutterBottom>
          Leave Analytics
        </Typography>
        <Typography variant="body1" color="textSecondary">
          Analytics and reporting features will be implemented here.
        </Typography>
      </TabPanel>
      <LeaveDialog
        open={dialogOpen}
        leave={selectedLeave}
        onClose={() => setDialogOpen(false)}
        onSave={handleSave}
        onApprove={handleApprove}
        onReject={handleReject}
        mode={dialogMode}
      />
    </Box>
  );
};