import React, { useState, useEffect } from 'react';
import {
  Box,
  Typography,
  Button,
  Card,
  CardContent,
  IconButton,
  Tabs,
  Tab,
  Alert,
  Snackbar,
  Fab,
  Menu,
  MenuItem,
  ListItemIcon,
  ListItemText,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogContentText,
  DialogActions,
  Chip,
  List,
  ListItem,
  ListItemAvatar,
  Avatar,
  ListItemSecondaryAction,
  Divider,
  LinearProgress,
} from '@mui/material';
import {
  Add,
  Event,
  CalendarToday,
  List as ListIcon,
  Edit,
  Delete,
  MoreVert,
  FileDownload,
  FileUpload,
  Refresh,
  Today,
} from '@mui/icons-material';
import { format, isAfter, isBefore, startOfDay } from 'date-fns';
import { Holiday, HolidayStats, HolidayFilter } from '../types';
import { holidayService } from '../services';
import { HolidayCalendar } from '../components/Holidays/HolidayCalendar';
import { HolidayDialog } from '../components/Holidays/HolidayDialog';

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

function TabPanel(props: TabPanelProps) {
  const { children, value, index, ...other } = props;
  return (
    <div
      role="tabpanel"
      hidden={value !== index}
      id={`holidays-tabpanel-${index}`}
      {...other}
    >
      {value === index && <Box>{children}</Box>}
    </div>
  );
}

export const Holidays: React.FC = () => {
  const [tabValue, setTabValue] = useState(0);
  const [holidays, setHolidays] = useState<Holiday[]>([]);
  const [stats, setStats] = useState<HolidayStats | null>(null);
  const [loading, setLoading] = useState(false);
  const [snackbar, setSnackbar] = useState({ 
    open: false, 
    message: '', 
    severity: 'success' as 'success' | 'error' | 'warning' | 'info' 
  });

  // Dialog states
  const [holidayDialog, setHolidayDialog] = useState(false);
  const [editingHoliday, setEditingHoliday] = useState<Holiday | null>(null);
  const [deleteDialog, setDeleteDialog] = useState(false);
  const [holidayToDelete, setHolidayToDelete] = useState<Holiday | null>(null);
  const [selectedDate, setSelectedDate] = useState<Date | null>(null);

  // Menu states
  const [actionMenu, setActionMenu] = useState<{
    anchor: HTMLElement | null;
    holiday: Holiday | null;
  }>({ anchor: null, holiday: null });

  const organizationId = 'org-1'; // In real app, get from context
  const currentYear = new Date().getFullYear();

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    setLoading(true);
    try {
      const [holidaysResult, statsResult] = await Promise.all([
        holidayService.getAllHolidays(
          { year: currentYear, organizationId },
          { page: 1, limit: 100, sortBy: 'date', sortOrder: 'asc' }
        ),
        holidayService.getHolidayStats({ year: currentYear, organizationId })
      ]);

      if (holidaysResult) {
        setHolidays(holidaysResult.holidays);
      }
      
      if (statsResult) {
        setStats(statsResult);
      }
    } catch (error) {
      showSnackbar('Failed to load holidays data', 'error');
      console.error('Load holidays error:', error);
    } finally {
      setLoading(false);
    }
  };

  const showSnackbar = (message: string, severity: 'success' | 'error' | 'warning' | 'info') => {
    setSnackbar({ open: true, message, severity });
  };

  const handleAddHoliday = (preselectedDate?: Date) => {
    setEditingHoliday(null);
    setSelectedDate(preselectedDate || null);
    setHolidayDialog(true);
  };

  const handleEditHoliday = (holiday: Holiday) => {
    setEditingHoliday(holiday);
    setSelectedDate(null);
    setHolidayDialog(true);
    handleCloseActionMenu();
  };

  const handleDeleteHoliday = (holiday: Holiday) => {
    setHolidayToDelete(holiday);
    setDeleteDialog(true);
    handleCloseActionMenu();
  };

  const confirmDeleteHoliday = async () => {
    if (!holidayToDelete) return;

    try {
      const success = await holidayService.deleteHoliday(holidayToDelete.id);
      if (success) {
        showSnackbar('Holiday deleted successfully', 'success');
        loadData();
      } else {
        showSnackbar('Failed to delete holiday', 'error');
      }
    } catch (error) {
      showSnackbar('Failed to delete holiday', 'error');
    }

    setDeleteDialog(false);
    setHolidayToDelete(null);
  };

  const handleSaveHoliday = (holiday: Holiday) => {
    showSnackbar(
      editingHoliday ? 'Holiday updated successfully' : 'Holiday added successfully',
      'success'
    );
    loadData();
  };

  const handleDateClick = (date: Date, dateHolidays: Holiday[]) => {
    if (dateHolidays.length === 0) {
      handleAddHoliday(date);
    }
  };

  const handleHolidayClick = (holiday: Holiday) => {
    handleEditHoliday(holiday);
  };

  const handleActionMenuOpen = (event: React.MouseEvent<HTMLElement>, holiday: Holiday) => {
    setActionMenu({ anchor: event.currentTarget, holiday });
  };

  const handleCloseActionMenu = () => {
    setActionMenu({ anchor: null, holiday: null });
  };

  const getHolidayTypeColor = (type: string): "default" | "primary" | "secondary" | "error" | "info" | "success" | "warning" => {
    switch (type) {
      case 'public': return 'error';
      case 'company': return 'primary';
      case 'optional': return 'warning';
      case 'religious': return 'secondary';
      default: return 'default';
    }
  };

  const isUpcoming = (date: string): boolean => {
    return isAfter(new Date(date), startOfDay(new Date()));
  };

  const isPast = (date: string): boolean => {
    return isBefore(new Date(date), startOfDay(new Date()));
  };

  const renderStatsCards = () => (
    <Box sx={{ display: 'flex', gap: 2, mb: 3, flexWrap: 'wrap' }}>
      <Card sx={{ flex: '1 1 200px', minWidth: 200 }}>
        <CardContent>
          <Typography color="textSecondary" gutterBottom>
            Total Holidays
          </Typography>
          <Typography variant="h4" component="div">
            {stats?.total || 0}
          </Typography>
        </CardContent>
      </Card>

      <Card sx={{ flex: '1 1 200px', minWidth: 200 }}>
        <CardContent>
          <Typography color="textSecondary" gutterBottom>
            Upcoming
          </Typography>
          <Typography variant="h4" component="div">
            {stats?.upcoming || 0}
          </Typography>
        </CardContent>
      </Card>

      <Card sx={{ flex: '1 1 200px', minWidth: 200 }}>
        <CardContent>
          <Typography color="textSecondary" gutterBottom>
            This Month
          </Typography>
          <Typography variant="h4" component="div">
            {stats?.thisMonth || 0}
          </Typography>
        </CardContent>
      </Card>

      <Card sx={{ flex: '1 1 200px', minWidth: 200 }}>
        <CardContent>
          <Typography color="textSecondary" gutterBottom>
            Recurring
          </Typography>
          <Typography variant="h4" component="div">
            {stats?.recurring || 0}
          </Typography>
        </CardContent>
      </Card>
    </Box>
  );

  const renderHolidaysList = () => (
    <Card>
      <CardContent>
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 2 }}>
          <Typography variant="h6">All Holidays ({currentYear})</Typography>
          <Box>
            <IconButton onClick={loadData} disabled={loading}>
              <Refresh />
            </IconButton>
            <Button
              variant="contained"
              startIcon={<Add />}
              onClick={() => handleAddHoliday()}
              sx={{ ml: 1 }}
            >
              Add Holiday
            </Button>
          </Box>
        </Box>

        {loading && <LinearProgress sx={{ mb: 2 }} />}

        <List>
          {holidays.map((holiday, index) => (
            <React.Fragment key={holiday.id}>
              <ListItem>
                <ListItemAvatar>
                  <Avatar sx={{ bgcolor: holiday.color || '#2196f3' }}>
                    <Event />
                  </Avatar>
                </ListItemAvatar>
                
                <ListItemText
                  primary={
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                      <Typography variant="subtitle1">{holiday.name}</Typography>
                      <Chip 
                        label={holiday.type} 
                        size="small" 
                        color={getHolidayTypeColor(holiday.type)}
                      />
                      {holiday.isRecurring && (
                        <Chip label="Recurring" size="small" variant="outlined" />
                      )}
                      {isPast(holiday.date) && (
                        <Chip label="Past" size="small" color="default" />
                      )}
                      {isUpcoming(holiday.date) && (
                        <Chip label="Upcoming" size="small" color="info" />
                      )}
                    </Box>
                  }
                  secondary={
                    <Box>
                      <Typography variant="body2" color="textPrimary">
                        {format(new Date(holiday.date), 'EEEE, MMMM do, yyyy')}
                      </Typography>
                      {holiday.description && (
                        <Typography variant="body2" color="textSecondary">
                          {holiday.description}
                        </Typography>
                      )}
                    </Box>
                  }
                />

                <ListItemSecondaryAction>
                  <IconButton
                    onClick={(e) => handleActionMenuOpen(e, holiday)}
                    edge="end"
                  >
                    <MoreVert />
                  </IconButton>
                </ListItemSecondaryAction>
              </ListItem>
              {index < holidays.length - 1 && <Divider />}
            </React.Fragment>
          ))}
        </List>

        {holidays.length === 0 && !loading && (
          <Box sx={{ textAlign: 'center', py: 4 }}>
            <Event sx={{ fontSize: 64, color: 'text.secondary', mb: 2 }} />
            <Typography variant="h6" color="textSecondary" gutterBottom>
              No holidays found
            </Typography>
            <Typography variant="body2" color="textSecondary" sx={{ mb: 2 }}>
              Add your first holiday to get started
            </Typography>
            <Button
              variant="contained"
              startIcon={<Add />}
              onClick={() => handleAddHoliday()}
            >
              Add Holiday
            </Button>
          </Box>
        )}
      </CardContent>
    </Card>
  );

  return (
    <Box sx={{ p: 3 }}>
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
        <Typography variant="h4" gutterBottom>
          Holidays Management
        </Typography>
        
        <Button
          variant="contained"
          startIcon={<Add />}
          onClick={() => handleAddHoliday()}
          sx={{ display: { xs: 'none', sm: 'flex' } }}
        >
          Add Holiday
        </Button>
      </Box>

      {renderStatsCards()}

      <Tabs
        value={tabValue}
        onChange={(_, newValue) => setTabValue(newValue)}
        indicatorColor="primary"
        textColor="primary"
        sx={{ mb: 3 }}
      >
        <Tab icon={<CalendarToday />} label="Calendar View" />
        <Tab icon={<ListIcon />} label="List View" />
      </Tabs>

      <TabPanel value={tabValue} index={0}>
        <HolidayCalendar
          organizationId={organizationId}
          onDateClick={handleDateClick}
          onHolidayClick={handleHolidayClick}
          height={600}
        />
      </TabPanel>

      <TabPanel value={tabValue} index={1}>
        {renderHolidaysList()}
      </TabPanel>

      {/* Floating Action Button for Mobile */}
      <Fab
        color="primary"
        aria-label="add holiday"
        sx={{ 
          position: 'fixed', 
          bottom: 16, 
          right: 16,
          display: { xs: 'flex', sm: 'none' }
        }}
        onClick={() => handleAddHoliday()}
      >
        <Add />
      </Fab>

      {/* Holiday Dialog */}
      <HolidayDialog
        open={holidayDialog}
        onClose={() => setHolidayDialog(false)}
        onSave={handleSaveHoliday}
        holiday={editingHoliday}
        organizationId={organizationId}
        preselectedDate={selectedDate}
      />

      {/* Action Menu */}
      <Menu
        anchorEl={actionMenu.anchor}
        open={Boolean(actionMenu.anchor)}
        onClose={handleCloseActionMenu}
      >
        <MenuItem onClick={() => actionMenu.holiday && handleEditHoliday(actionMenu.holiday)}>
          <ListItemIcon>
            <Edit fontSize="small" />
          </ListItemIcon>
          <ListItemText>Edit</ListItemText>
        </MenuItem>
        <MenuItem onClick={() => actionMenu.holiday && handleDeleteHoliday(actionMenu.holiday)}>
          <ListItemIcon>
            <Delete fontSize="small" />
          </ListItemIcon>
          <ListItemText>Delete</ListItemText>
        </MenuItem>
      </Menu>

      {/* Delete Confirmation Dialog */}
      <Dialog
        open={deleteDialog}
        onClose={() => setDeleteDialog(false)}
      >
        <DialogTitle>Delete Holiday</DialogTitle>
        <DialogContent>
          <DialogContentText>
            Are you sure you want to delete "{holidayToDelete?.name}"? 
            This action cannot be undone.
          </DialogContentText>
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setDeleteDialog(false)}>Cancel</Button>
          <Button onClick={confirmDeleteHoliday} color="error" variant="contained">
            Delete
          </Button>
        </DialogActions>
      </Dialog>

      {/* Snackbar */}
      <Snackbar
        open={snackbar.open}
        autoHideDuration={6000}
        onClose={() => setSnackbar({ ...snackbar, open: false })}
      >
        <Alert 
          severity={snackbar.severity} 
          onClose={() => setSnackbar({ ...snackbar, open: false })}
        >
          {snackbar.message}
        </Alert>
      </Snackbar>
    </Box>
  );
};