import React, { useState } from 'react';
import {
  Box,
  Container,
  Typography,
  Tabs,
  Tab,
  Paper,
  Breadcrumbs,
  Link,
  Chip,
} from '@mui/material';
import {
  Policy,
  AccountBalance,
  Home,
  Assessment,
} from '@mui/icons-material';
import { PolicyManagement, ComplianceTracker } from '../components/Compliance';

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

function TabPanel(props: TabPanelProps) {
  const { children, value, index, ...other } = props;

  return (
    <div
      role="tabpanel"
      hidden={value !== index}
      id={`compliance-tabpanel-${index}`}
      aria-labelledby={`compliance-tab-${index}`}
      {...other}
    >
      {value === index && <Box>{children}</Box>}
    </div>
  );
}

function a11yProps(index: number) {
  return {
    id: `compliance-tab-${index}`,
    'aria-controls': `compliance-tabpanel-${index}`,
  };
}

export const Compliance: React.FC = () => {
  const [tabValue, setTabValue] = useState(0);

  const handleTabChange = (_: React.SyntheticEvent, newValue: number) => {
    setTabValue(newValue);
  };

  return (
    <Container maxWidth="xl" sx={{ mt: 2, mb: 4 }}>
      {/* Header */}
      <Box sx={{ mb: 3 }}>
        <Breadcrumbs aria-label="breadcrumb" sx={{ mb: 2 }}>
          <Link
            color="inherit"
            href="/dashboard"
            sx={{ display: 'flex', alignItems: 'center' }}
          >
            <Home sx={{ mr: 0.5 }} fontSize="inherit" />
            Dashboard
          </Link>
          <Typography
            color="text.primary"
            sx={{ display: 'flex', alignItems: 'center' }}
          >
            <Assessment sx={{ mr: 0.5 }} fontSize="inherit" />
            Compliance & Policies
          </Typography>
        </Breadcrumbs>

        <Box sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between', mb: 2 }}>
          <Box>
            <Typography variant="h4" component="h1" gutterBottom>
              Compliance & Policies
            </Typography>
            <Typography variant="body1" color="text.secondary">
              Manage HR policies and legal compliance (PF, ESI, Gratuity)
            </Typography>
          </Box>
          
          <Box sx={{ display: 'flex', gap: 1 }}>
            <Chip
              icon={<Policy />}
              label="Policy Management"
              color="primary"
              variant={tabValue === 0 ? "filled" : "outlined"}
              onClick={() => setTabValue(0)}
              clickable
            />
            <Chip
              icon={<AccountBalance />}
              label="Legal Compliance"
              color="secondary"
              variant={tabValue === 1 ? "filled" : "outlined"}
              onClick={() => setTabValue(1)}
              clickable
            />
          </Box>
        </Box>
      </Box>

      {/* Content */}
      <Paper sx={{ width: '100%' }}>
        <Box sx={{ borderBottom: 1, borderColor: 'divider' }}>
          <Tabs
            value={tabValue}
            onChange={handleTabChange}
            aria-label="compliance tabs"
            variant="fullWidth"
          >
            <Tab
              icon={<Policy />}
              label="HR Policies"
              {...a11yProps(0)}
              sx={{ minHeight: 72 }}
            />
            <Tab
              icon={<AccountBalance />}
              label="Legal Compliance"
              {...a11yProps(1)}
              sx={{ minHeight: 72 }}
            />
          </Tabs>
        </Box>

        <TabPanel value={tabValue} index={0}>
          <Box sx={{ p: 0 }}>
            <PolicyManagement />
          </Box>
        </TabPanel>

        <TabPanel value={tabValue} index={1}>
          <Box sx={{ p: 0 }}>
            <ComplianceTracker />
          </Box>
        </TabPanel>
      </Paper>

      {/* Footer Information */}
      <Box sx={{ mt: 4, p: 3, bgcolor: 'background.paper', borderRadius: 2 }}>
        <Typography variant="h6" gutterBottom>
          About Compliance & Policies Module
        </Typography>
        
        <Box sx={{ display: 'grid', gridTemplateColumns: { xs: '1fr', md: '1fr 1fr' }, gap: 3, mt: 2 }}>
          <Box>
            <Typography variant="subtitle1" color="primary" gutterBottom>
              HR Policies
            </Typography>
            <Typography variant="body2" color="text.secondary" paragraph>
              Maintain and manage all HR policies including attendance, leave, code of conduct, 
              disciplinary actions, benefits, and safety policies. Track employee acknowledgments 
              and ensure policy compliance across the organization.
            </Typography>
            <Typography variant="body2" color="text.secondary">
              Features: Policy creation, version control, employee acknowledgments, 
              categorization, and automated notifications.
            </Typography>
          </Box>
          
          <Box>
            <Typography variant="subtitle1" color="secondary" gutterBottom>
              Legal Compliance
            </Typography>
            <Typography variant="body2" color="text.secondary" paragraph>
              Automate legal compliance calculations for Provident Fund (PF), Employee State Insurance (ESI), 
              Gratuity, Tax Deducted at Source (TDS), and Professional Tax (PT). Generate compliance 
              reports and ensure adherence to Indian labor laws.
            </Typography>
            <Typography variant="body2" color="text.secondary">
              Features: Automated calculations, compliance tracking, statutory reports, 
              audit trails, and regulatory updates.
            </Typography>
          </Box>
        </Box>
        
        <Box sx={{ mt: 3, p: 2, bgcolor: 'warning.light', borderRadius: 1 }}>
          <Typography variant="body2" color="warning.dark">
            <strong>Important:</strong> Compliance calculations are based on current Indian labor laws. 
            Please consult with legal and financial advisors to ensure accuracy and compliance with 
            the latest regulations. Regular updates to compliance rules may be required.
          </Typography>
        </Box>
      </Box>
    </Container>
  );
};