import React, { useState, useEffect } from 'react';
import {
  Box,
  Paper,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TablePagination,
  TableRow,
  Typography,
  Button,
  TextField,
  InputAdornment,
  Chip,
  Card,
  CardContent,
  Grid,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Avatar,
  IconButton,
  Tabs,
  Tab,
  Badge,
} from '@mui/material';
import {
  Search,
  CheckCircle,
  Cancel,
  Schedule,
  AccessTime,
  CalendarToday,
  TrendingUp,
  Add,
  Edit,
} from '@mui/icons-material';
import { Attendance } from '../types';
import { format, startOfWeek, endOfWeek, eachDayOfInterval } from 'date-fns';
import { useFormValidation, VALIDATION_SCHEMAS } from '../utils/validation';

const mockAttendance: Attendance[] = [
  {
    id: '1',
    employeeId: '1',
    date: '2023-06-19',
    checkIn: '09:00',
    checkOut: '17:30',
    status: 'present',
    workingHours: 8,
    overtime: 0.5,
  },
  {
    id: '2',
    employeeId: '2',
    date: '2023-06-19',
    checkIn: '09:15',
    checkOut: '17:45',
    status: 'late',
    workingHours: 8,
    overtime: 0.5,
  },
  {
    id: '3',
    employeeId: '3',
    date: '2023-06-19',
    checkIn: '09:00',
    checkOut: '13:00',
    status: 'half-day',
    workingHours: 4,
    overtime: 0,
  },
  {
    id: '4',
    employeeId: '4',
    date: '2023-06-19',
    status: 'absent',
    workingHours: 0,
    overtime: 0,
  },
  {
    id: '5',
    employeeId: '5',
    date: '2023-06-19',
    checkIn: '09:00',
    checkOut: '17:30',
    status: 'present',
    workingHours: 8,
    overtime: 0.5,
  },
];

const mockEmployeeNames: { [key: string]: string } = {
  '1': 'John Doe',
  '2': 'Sarah Johnson',
  '3': 'Michael Brown',
  '4': 'Emily Davis',
  '5': 'David Wilson',
};

interface AttendanceDialogProps {
  open: boolean;
  attendance: Attendance | null;
  onClose: () => void;
  onSave: (attendance: Partial<Attendance>) => void;
  mode: 'edit' | 'create';
}

const AttendanceDialog: React.FC<AttendanceDialogProps> = ({
  open,
  attendance,
  onClose,
  onSave,
  mode,
}) => {
  const [formData, setFormData] = useState<Partial<Attendance>>({
    employeeId: '',
    date: format(new Date(), 'yyyy-MM-dd'),
    checkIn: '',
    checkOut: '',
    status: 'present',
    workingHours: 8,
    overtime: 0,
    notes: '',
  });

  const {
    errors,
    validateForm,
    handleFieldChange,
    clearErrors,
    hasError,
    getErrorMessage,
  } = useFormValidation(VALIDATION_SCHEMAS.ATTENDANCE);

  useEffect(() => {
    if (attendance) {
      setFormData(attendance);
    } else if (mode === 'create') {
      setFormData({
        employeeId: '',
        date: format(new Date(), 'yyyy-MM-dd'),
        checkIn: '',
        checkOut: '',
        status: 'present',
        workingHours: 8,
        overtime: 0,
        notes: '',
      });
    }
  }, [attendance, mode]);

  const handleFieldChangeWithValidation = (fieldName: string, value: any) => {
    setFormData(prev => ({ ...prev, [fieldName]: value }));
    handleFieldChange(fieldName, value);
  };

  const handleSubmit = () => {
    const validation = validateForm(formData);
    if (!validation.isValid) {
      return;
    }
    onSave(formData);
    clearErrors();
  };

  return (
    <Dialog open={open} onClose={onClose} maxWidth="sm" fullWidth>
      <DialogTitle>
        {mode === 'create' ? 'Add Attendance Record' : 'Edit Attendance'}
      </DialogTitle>
      <DialogContent>
        <Grid container spacing={2} sx={{ mt: 1 }}>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Employee ID"
              value={formData.employeeId || ''}
              onChange={(e) => handleFieldChangeWithValidation('employeeId', e.target.value)}
              required
              error={hasError('employeeId')}
              helperText={getErrorMessage('employeeId')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Date"
              type="date"
              value={formData.date || ''}
              onChange={(e) => handleFieldChangeWithValidation('date', e.target.value)}
              InputLabelProps={{ shrink: true }}
              required
              error={hasError('date')}
              helperText={getErrorMessage('date')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Check In"
              type="time"
              value={formData.checkIn || ''}
              onChange={(e) => handleFieldChangeWithValidation('checkIn', e.target.value)}
              InputLabelProps={{ shrink: true }}
              error={hasError('checkIn')}
              helperText={getErrorMessage('checkIn')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Check Out"
              type="time"
              value={formData.checkOut || ''}
              onChange={(e) => handleFieldChangeWithValidation('checkOut', e.target.value)}
              InputLabelProps={{ shrink: true }}
              error={hasError('checkOut')}
              helperText={getErrorMessage('checkOut')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <FormControl fullWidth>
              <InputLabel>Status</InputLabel>
              <Select
                value={formData.status || 'present'}
                onChange={(e) => setFormData({ ...formData, status: e.target.value as any })}
              >
                <MenuItem value="present">Present</MenuItem>
                <MenuItem value="absent">Absent</MenuItem>
                <MenuItem value="late">Late</MenuItem>
                <MenuItem value="half-day">Half Day</MenuItem>
              </Select>
            </FormControl>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Working Hours"
              type="number"
              value={formData.workingHours || ''}
              onChange={(e) => handleFieldChangeWithValidation('workingHours', Number(e.target.value))}
              inputProps={{ step: 0.5, min: 0, max: 24 }}
              error={hasError('workingHours')}
              helperText={getErrorMessage('workingHours')}
            />
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6
            }}>
            <TextField
              fullWidth
              label="Overtime Hours"
              type="number"
              value={formData.overtime || ''}
              onChange={(e) => handleFieldChangeWithValidation('overtime', Number(e.target.value))}
              inputProps={{ step: 0.5, min: 0 }}
              error={hasError('overtime')}
              helperText={getErrorMessage('overtime')}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Notes"
              multiline
              rows={2}
              value={formData.notes || ''}
              onChange={(e) => handleFieldChangeWithValidation('notes', e.target.value)}
              error={hasError('notes')}
              helperText={getErrorMessage('notes')}
            />
          </Grid>
        </Grid>
      </DialogContent>
      <DialogActions>
        <Button onClick={onClose}>Cancel</Button>
        <Button onClick={handleSubmit} variant="contained">
          {mode === 'create' ? 'Add Record' : 'Save Changes'}
        </Button>
      </DialogActions>
    </Dialog>
  );
};

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

const TabPanel: React.FC<TabPanelProps> = ({ children, value, index }) => (
  <div hidden={value !== index}>
    {value === index && <Box sx={{ py: 3 }}>{children}</Box>}
  </div>
);

export const AttendancePage: React.FC = () => {
  const [attendance, setAttendance] = useState<Attendance[]>(mockAttendance);
  const [filteredAttendance, setFilteredAttendance] = useState<Attendance[]>(mockAttendance);
  const [searchTerm, setSearchTerm] = useState('');
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [selectedAttendance, setSelectedAttendance] = useState<Attendance | null>(null);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [dialogMode, setDialogMode] = useState<'edit' | 'create'>('create');
  const [tabValue, setTabValue] = useState(0);
  const [selectedDate, setSelectedDate] = useState(format(new Date(), 'yyyy-MM-dd'));

  useEffect(() => {
    filterAttendance();
  }, [attendance, searchTerm, selectedDate]);

  const filterAttendance = () => {
    let filtered = attendance.filter(att => att.date === selectedDate);

    if (searchTerm) {
      filtered = filtered.filter(att =>
        mockEmployeeNames[att.employeeId]?.toLowerCase().includes(searchTerm.toLowerCase()) ||
        att.employeeId.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    setFilteredAttendance(filtered);
    setPage(0);
  };

  const handleEdit = (attendance: Attendance) => {
    setSelectedAttendance(attendance);
    setDialogMode('edit');
    setDialogOpen(true);
  };

  const handleCreate = () => {
    setSelectedAttendance(null);
    setDialogMode('create');
    setDialogOpen(true);
  };

  const handleSave = (attendanceData: Partial<Attendance>) => {
    if (dialogMode === 'create') {
      const newAttendance: Attendance = {
        ...attendanceData,
        id: Date.now().toString(),
      } as Attendance;
      setAttendance(prev => [...prev, newAttendance]);
    } else if (dialogMode === 'edit' && selectedAttendance) {
      setAttendance(prev =>
        prev.map(att => att.id === selectedAttendance.id ? { ...att, ...attendanceData } : att)
      );
    }
    setDialogOpen(false);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'present': return 'success';
      case 'absent': return 'error';
      case 'late': return 'warning';
      case 'half-day': return 'info';
      default: return 'default';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'present': return <CheckCircle />;
      case 'absent': return <Cancel />;
      case 'late': return <AccessTime />;
      case 'half-day': return <Schedule />;
      default: return <Schedule />;
    }
  };

  const todayStats = {
    total: filteredAttendance.length,
    present: filteredAttendance.filter(att => att.status === 'present').length,
    absent: filteredAttendance.filter(att => att.status === 'absent').length,
    late: filteredAttendance.filter(att => att.status === 'late').length,
    halfDay: filteredAttendance.filter(att => att.status === 'half-day').length,
  };

  const attendanceRate = todayStats.total > 0 
    ? Math.round(((todayStats.present + todayStats.late + todayStats.halfDay) / todayStats.total) * 100)
    : 0;

  return (
    <Box>
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
        <Typography variant="h4" gutterBottom>
          Attendance Management
        </Typography>
        <Button variant="contained" startIcon={<Add />} onClick={handleCreate}>
          Add Record
        </Button>
      </Box>
      <Box sx={{ borderBottom: 1, borderColor: 'divider', mb: 3 }}>
        <Tabs value={tabValue} onChange={(e, newValue) => setTabValue(newValue)}>
          <Tab label="Daily View" />
          <Tab label="Weekly View" />
          <Tab label="Monthly Summary" />
        </Tabs>
      </Box>
      <TabPanel value={tabValue} index={0}>
        {/* Daily Stats Cards */}
        <Grid container spacing={3} sx={{ mb: 3 }}>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 2.4
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <Typography color="textSecondary" gutterBottom>
                  Total
                </Typography>
                <Typography variant="h4">
                  {todayStats.total}
                </Typography>
              </CardContent>
            </Card>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 2.4
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <Badge badgeContent={todayStats.present} color="success">
                  <CheckCircle color="success" sx={{ fontSize: 24 }} />
                </Badge>
                <Typography variant="h6" sx={{ mt: 1 }}>
                  Present
                </Typography>
              </CardContent>
            </Card>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 2.4
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <Badge badgeContent={todayStats.absent} color="error">
                  <Cancel color="error" sx={{ fontSize: 24 }} />
                </Badge>
                <Typography variant="h6" sx={{ mt: 1 }}>
                  Absent
                </Typography>
              </CardContent>
            </Card>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 2.4
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <Badge badgeContent={todayStats.late} color="warning">
                  <AccessTime color="warning" sx={{ fontSize: 24 }} />
                </Badge>
                <Typography variant="h6" sx={{ mt: 1 }}>
                  Late
                </Typography>
              </CardContent>
            </Card>
          </Grid>
          <Grid
            size={{
              xs: 12,
              sm: 6,
              md: 2.4
            }}>
            <Card>
              <CardContent sx={{ textAlign: 'center' }}>
                <Typography color="textSecondary" gutterBottom>
                  Attendance Rate
                </Typography>
                <Typography variant="h4" color="primary">
                  {attendanceRate}%
                </Typography>
              </CardContent>
            </Card>
          </Grid>
        </Grid>

        {/* Filters */}
        <Paper sx={{ mb: 2, p: 2 }}>
          <Box sx={{ display: 'flex', gap: 2, alignItems: 'center', flexWrap: 'wrap' }}>
            <TextField
              size="small"
              placeholder="Search employees..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              InputProps={{
                startAdornment: (
                  <InputAdornment position="start">
                    <Search />
                  </InputAdornment>
                ),
              }}
              sx={{ minWidth: 250 }}
            />
            <TextField
              size="small"
              type="date"
              label="Date"
              value={selectedDate}
              onChange={(e) => setSelectedDate(e.target.value)}
              InputLabelProps={{ shrink: true }}
            />
          </Box>
        </Paper>

        {/* Attendance Table */}
        <TableContainer component={Paper}>
          <Table>
            <TableHead>
              <TableRow>
                <TableCell>Employee</TableCell>
                <TableCell>Date</TableCell>
                <TableCell>Check In</TableCell>
                <TableCell>Check Out</TableCell>
                <TableCell>Status</TableCell>
                <TableCell>Working Hours</TableCell>
                <TableCell>Overtime</TableCell>
                <TableCell>Actions</TableCell>
              </TableRow>
            </TableHead>
            <TableBody>
              {filteredAttendance
                .slice(page * rowsPerPage, page * rowsPerPage + rowsPerPage)
                .map((att) => (
                  <TableRow key={att.id} hover>
                    <TableCell>
                      <Box sx={{ display: 'flex', alignItems: 'center', gap: 2 }}>
                        <Avatar sx={{ width: 32, height: 32 }}>
                          {mockEmployeeNames[att.employeeId]?.split(' ').map(n => n[0]).join('')}
                        </Avatar>
                        <Box>
                          <Typography variant="body2" fontWeight="bold">
                            {mockEmployeeNames[att.employeeId] || `Employee ${att.employeeId}`}
                          </Typography>
                          <Typography variant="body2" color="textSecondary">
                            ID: {att.employeeId}
                          </Typography>
                        </Box>
                      </Box>
                    </TableCell>
                    <TableCell>{format(new Date(att.date), 'MMM dd, yyyy')}</TableCell>
                    <TableCell>{att.checkIn || '-'}</TableCell>
                    <TableCell>{att.checkOut || '-'}</TableCell>
                    <TableCell>
                      <Chip
                        icon={getStatusIcon(att.status)}
                        label={att.status}
                        color={getStatusColor(att.status) as any}
                        size="small"
                      />
                    </TableCell>
                    <TableCell>{att.workingHours}h</TableCell>
                    <TableCell>{att.overtime}h</TableCell>
                    <TableCell>
                      <IconButton onClick={() => handleEdit(att)} size="small">
                        <Edit />
                      </IconButton>
                    </TableCell>
                  </TableRow>
                ))}
            </TableBody>
          </Table>
          <TablePagination
            rowsPerPageOptions={[5, 10, 25]}
            component="div"
            count={filteredAttendance.length}
            rowsPerPage={rowsPerPage}
            page={page}
            onPageChange={(event, newPage) => setPage(newPage)}
            onRowsPerPageChange={(event) => {
              setRowsPerPage(parseInt(event.target.value, 10));
              setPage(0);
            }}
          />
        </TableContainer>
      </TabPanel>
      <TabPanel value={tabValue} index={1}>
        <Typography variant="h6" gutterBottom>
          Weekly Attendance Overview
        </Typography>
        <Typography variant="body1" color="textSecondary">
          Weekly view functionality will be implemented here.
        </Typography>
      </TabPanel>
      <TabPanel value={tabValue} index={2}>
        <Typography variant="h6" gutterBottom>
          Monthly Summary
        </Typography>
        <Typography variant="body1" color="textSecondary">
          Monthly summary and analytics will be implemented here.
        </Typography>
      </TabPanel>
      <AttendanceDialog
        open={dialogOpen}
        attendance={selectedAttendance}
        onClose={() => setDialogOpen(false)}
        onSave={handleSave}
        mode={dialogMode}
      />
    </Box>
  );
};