export interface AppConfig {
  app: {
    name: string;
    version: string;
    description: string;
  };
  api: {
    baseUrl: string;
    timeout: number;
    version: string;
  };
  auth: {
    jwtSecret: string;
    jwtExpiresIn: string;
    sessionTimeout: number;
  };
  features: {
    notifications: boolean;
    darkMode: boolean;
    mobileView: boolean;
    reports: boolean;
    compliance: boolean;
  };
  security: {
    enable2FA: boolean;
    passwordMinLength: number;
    loginAttemptsLimit: number;
    accountLockoutTime: number;
  };
  company: {
    name: string;
    address: string;
    phone: string;
    email: string;
    website: string;
  };
  compliance: {
    pfRate: number;
    esiRateEmployee: number;
    esiRateEmployer: number;
    professionalTax: number;
    gratuityYearsMin: number;
  };
  settings: {
    defaultCurrency: string;
    dateFormat: string;
    timeFormat: string;
    timezone: string;
  };
  upload: {
    maxFileSize: number;
    allowedFileTypes: string[];
    uploadEndpoint: string;
  };
  email: {
    smtpHost: string;
    smtpPort: number;
    smtpSecure: boolean;
    smtpUser: string;
    smtpPass: string;
    emailFrom: string;
    emailFromName: string;
  };
  development: {
    debug: boolean;
    logLevel: string;
  };
}

const getEnvVar = (key: string, defaultValue: string = ''): string => {
  return process.env[key] || defaultValue;
};

const getEnvBoolean = (key: string, defaultValue: boolean = false): boolean => {
  const value = process.env[key];
  if (value === undefined) return defaultValue;
  return value.toLowerCase() === 'true';
};

const getEnvNumber = (key: string, defaultValue: number = 0): number => {
  const value = process.env[key];
  if (value === undefined) return defaultValue;
  return parseInt(value, 10) || defaultValue;
};

export const appConfig: AppConfig = {
  app: {
    name: getEnvVar('REACT_APP_NAME', 'HR Management System'),
    version: getEnvVar('REACT_APP_VERSION', '1.0.0'),
    description: getEnvVar('REACT_APP_DESCRIPTION', 'Human Resources Management Solution'),
  },
  api: {
    baseUrl: getEnvVar('REACT_APP_API_BASE_URL', 'http://localhost:3001/api'),
    timeout: getEnvNumber('REACT_APP_API_TIMEOUT', 30000),
    version: getEnvVar('REACT_APP_API_VERSION', 'v1'),
  },
  auth: {
    jwtSecret: getEnvVar('REACT_APP_JWT_SECRET', 'default-secret'),
    jwtExpiresIn: getEnvVar('REACT_APP_JWT_EXPIRES_IN', '24h'),
    sessionTimeout: getEnvNumber('REACT_APP_SESSION_TIMEOUT', 3600000),
  },
  features: {
    notifications: getEnvBoolean('REACT_APP_ENABLE_NOTIFICATIONS', true),
    darkMode: getEnvBoolean('REACT_APP_ENABLE_DARK_MODE', true),
    mobileView: getEnvBoolean('REACT_APP_ENABLE_MOBILE_VIEW', true),
    reports: getEnvBoolean('REACT_APP_ENABLE_REPORTS', true),
    compliance: getEnvBoolean('REACT_APP_ENABLE_COMPLIANCE', true),
  },
  security: {
    enable2FA: getEnvBoolean('REACT_APP_ENABLE_2FA', false),
    passwordMinLength: getEnvNumber('REACT_APP_PASSWORD_MIN_LENGTH', 8),
    loginAttemptsLimit: getEnvNumber('REACT_APP_LOGIN_ATTEMPTS_LIMIT', 5),
    accountLockoutTime: getEnvNumber('REACT_APP_ACCOUNT_LOCKOUT_TIME', 1800000),
  },
  company: {
    name: getEnvVar('REACT_APP_COMPANY_NAME', 'Your Company Name'),
    address: getEnvVar('REACT_APP_COMPANY_ADDRESS', '123 Business Street, City, State 12345'),
    phone: getEnvVar('REACT_APP_COMPANY_PHONE', '+1 (555) 123-4567'),
    email: getEnvVar('REACT_APP_COMPANY_EMAIL', 'info@company.com'),
    website: getEnvVar('REACT_APP_COMPANY_WEBSITE', 'https://www.company.com'),
  },
  compliance: {
    pfRate: getEnvNumber('REACT_APP_PF_RATE', 12),
    esiRateEmployee: getEnvNumber('REACT_APP_ESI_RATE_EMPLOYEE', 0.75),
    esiRateEmployer: getEnvNumber('REACT_APP_ESI_RATE_EMPLOYER', 3.25),
    professionalTax: getEnvNumber('REACT_APP_PROFESSIONAL_TAX', 200),
    gratuityYearsMin: getEnvNumber('REACT_APP_GRATUITY_YEARS_MIN', 5),
  },
  settings: {
    defaultCurrency: getEnvVar('REACT_APP_DEFAULT_CURRENCY', 'INR'),
    dateFormat: getEnvVar('REACT_APP_DATE_FORMAT', 'DD/MM/YYYY'),
    timeFormat: getEnvVar('REACT_APP_TIME_FORMAT', '24h'),
    timezone: getEnvVar('REACT_APP_TIMEZONE', 'Asia/Kolkata'),
  },
  upload: {
    maxFileSize: getEnvNumber('REACT_APP_MAX_FILE_SIZE', 10485760), // 10MB
    allowedFileTypes: getEnvVar('REACT_APP_ALLOWED_FILE_TYPES', '.pdf,.doc,.docx,.jpg,.jpeg,.png,.xlsx,.xls').split(','),
    uploadEndpoint: getEnvVar('REACT_APP_UPLOAD_ENDPOINT', '/api/upload'),
  },
  email: {
    smtpHost: getEnvVar('SMTP_HOST', 'smtp.gmail.com'),
    smtpPort: getEnvNumber('SMTP_PORT', 587),
    smtpSecure: getEnvBoolean('SMTP_SECURE', false),
    smtpUser: getEnvVar('SMTP_USER', ''),
    smtpPass: getEnvVar('SMTP_PASS', ''),
    emailFrom: getEnvVar('EMAIL_FROM', 'noreply@company.com'),
    emailFromName: getEnvVar('EMAIL_FROM_NAME', 'HR Management System'),
  },
  development: {
    debug: getEnvBoolean('REACT_APP_DEBUG', true),
    logLevel: getEnvVar('REACT_APP_LOG_LEVEL', 'info'),
  },
};

export default appConfig;