import React, { useState, useEffect } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  TextField,
  Box,
  Typography,
  IconButton,
  Alert,
  CircularProgress,
  Accordion,
  AccordionSummary,
  AccordionDetails,
  FormControlLabel,
  Checkbox,
  Switch,
  Chip,
} from '@mui/material';
import {
  Close,
  Save,
  ExpandMore,
  Security,
  People,
  Work,
  Assessment,
  Settings,
} from '@mui/icons-material';
import { Role, RoleCreateRequest, RoleUpdateRequest, PermissionGroup } from '../../types';
import { userManagementService } from '../../services';

interface RoleDialogProps {
  open: boolean;
  onClose: () => void;
  onSave: (role: Role) => void;
  role?: Role | null;
}

export const RoleDialog: React.FC<RoleDialogProps> = ({
  open,
  onClose,
  onSave,
  role,
}) => {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const [permissionGroups, setPermissionGroups] = useState<PermissionGroup[]>([]);
  const [formData, setFormData] = useState<{
    name: string;
    description: string;
    selectedPermissions: string[];
    isActive: boolean;
  }>({
    name: '',
    description: '',
    selectedPermissions: [],
    isActive: true,
  });

  useEffect(() => {
    if (open) {
      loadPermissionGroups();
    }
  }, [open]);

  useEffect(() => {
    if (role) {
      setFormData({
        name: role.name,
        description: role.description,
        selectedPermissions: role.permissions.map(p => p.id),
        isActive: role.isActive,
      });
    } else {
      setFormData({
        name: '',
        description: '',
        selectedPermissions: [],
        isActive: true,
      });
    }
    setError('');
  }, [role, open]);

  const loadPermissionGroups = async () => {
    try {
      const groups = await userManagementService.getPermissionGroups();
      setPermissionGroups(groups);
    } catch (error) {
      console.error('Failed to load permission groups:', error);
    }
  };

  const handleInputChange = (field: string, value: any) => {
    setFormData({ ...formData, [field]: value });
  };

  const handlePermissionToggle = (permissionId: string) => {
    const currentPermissions = [...formData.selectedPermissions];
    const index = currentPermissions.indexOf(permissionId);
    
    if (index > -1) {
      currentPermissions.splice(index, 1);
    } else {
      currentPermissions.push(permissionId);
    }
    
    setFormData({ ...formData, selectedPermissions: currentPermissions });
  };

  const handleGroupToggle = (group: PermissionGroup) => {
    const groupPermissionIds = group.permissions.map(p => p.id);
    const areAllSelected = groupPermissionIds.every(id => 
      formData.selectedPermissions.includes(id)
    );
    
    let newPermissions = [...formData.selectedPermissions];
    
    if (areAllSelected) {
      // Remove all group permissions
      newPermissions = newPermissions.filter(id => !groupPermissionIds.includes(id));
    } else {
      // Add all group permissions
      groupPermissionIds.forEach(id => {
        if (!newPermissions.includes(id)) {
          newPermissions.push(id);
        }
      });
    }
    
    setFormData({ ...formData, selectedPermissions: newPermissions });
  };

  const isGroupSelected = (group: PermissionGroup) => {
    const groupPermissionIds = group.permissions.map(p => p.id);
    return groupPermissionIds.every(id => formData.selectedPermissions.includes(id));
  };

  const isGroupPartiallySelected = (group: PermissionGroup) => {
    const groupPermissionIds = group.permissions.map(p => p.id);
    const selectedInGroup = groupPermissionIds.filter(id => 
      formData.selectedPermissions.includes(id)
    );
    return selectedInGroup.length > 0 && selectedInGroup.length < groupPermissionIds.length;
  };

  const handleSave = async () => {
    setLoading(true);
    setError('');

    try {
      let result: Role | null;

      if (role) {
        // Update role
        const updateData: RoleUpdateRequest = {
          name: formData.name,
          description: formData.description,
          permissionIds: formData.selectedPermissions,
          isActive: formData.isActive,
        };
        
        result = await userManagementService.updateRole(role.id, updateData);
      } else {
        // Create role
        const createData: RoleCreateRequest = {
          name: formData.name,
          description: formData.description,
          permissionIds: formData.selectedPermissions,
        };

        // Validate role data
        const validation = await userManagementService.validateRole(createData);
        if (!validation.isValid) {
          setError(validation.errors.join(', '));
          setLoading(false);
          return;
        }

        result = await userManagementService.createRole(createData);
      }

      if (result) {
        onSave(result);
        onClose();
      } else {
        setError('Failed to save role');
      }
    } catch (err: any) {
      setError(err.message || 'Failed to save role');
    } finally {
      setLoading(false);
    }
  };

  const isFormValid = () => {
    return formData.name.trim().length >= 2 && formData.selectedPermissions.length > 0;
  };

  const getCategoryIcon = (category: string) => {
    switch (category) {
      case 'system': return <Settings />;
      case 'hr': return <People />;
      case 'employee': return <Work />;
      case 'finance': return <Assessment />;
      default: return <Security />;
    }
  };

  const getCategoryColor = (category: string): "default" | "primary" | "secondary" | "error" | "info" | "success" | "warning" => {
    switch (category) {
      case 'system': return 'error';
      case 'hr': return 'primary';
      case 'employee': return 'info';
      case 'finance': return 'success';
      case 'reports': return 'warning';
      default: return 'default';
    }
  };

  return (
    <Dialog
      open={open}
      onClose={onClose}
      maxWidth="md"
      fullWidth
      PaperProps={{
        sx: { minHeight: '80vh', maxHeight: '90vh' }
      }}
    >
      <DialogTitle sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
        <Typography variant="h6">
          {role ? 'Edit Role' : 'Create New Role'}
        </Typography>
        <IconButton onClick={onClose}>
          <Close />
        </IconButton>
      </DialogTitle>

      <DialogContent sx={{ overflow: 'auto' }}>
        {error && (
          <Alert severity="error" sx={{ mb: 2 }}>
            {error}
          </Alert>
        )}

        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 3, pt: 2 }}>
          {/* Basic Information */}
          <Box>
            <Typography variant="subtitle1" gutterBottom fontWeight="medium">
              Basic Information
            </Typography>
            
            <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
              <TextField
                label="Role Name"
                value={formData.name}
                onChange={(e) => handleInputChange('name', e.target.value)}
                fullWidth
                required
                disabled={role?.isSystem}
                helperText={role?.isSystem ? 'System roles cannot be renamed' : ''}
              />

              <TextField
                label="Description"
                value={formData.description}
                onChange={(e) => handleInputChange('description', e.target.value)}
                multiline
                rows={3}
                fullWidth
                placeholder="Describe what this role can do..."
              />

              <FormControlLabel
                control={
                  <Switch
                    checked={formData.isActive}
                    onChange={(e) => handleInputChange('isActive', e.target.checked)}
                  />
                }
                label="Active Role"
              />
            </Box>
          </Box>

          {/* Permissions */}
          <Box>
            <Typography variant="subtitle1" gutterBottom fontWeight="medium">
              Permissions ({formData.selectedPermissions.length} selected)
            </Typography>
            
            {permissionGroups.map((group) => (
              <Accordion key={group.id} sx={{ mb: 1 }}>
                <AccordionSummary expandIcon={<ExpandMore />}>
                  <Box sx={{ display: 'flex', alignItems: 'center', gap: 2, width: '100%' }}>
                    <FormControlLabel
                      control={
                        <Checkbox
                          checked={isGroupSelected(group)}
                          indeterminate={isGroupPartiallySelected(group)}
                          onChange={() => handleGroupToggle(group)}
                          onClick={(e) => e.stopPropagation()}
                        />
                      }
                      label=""
                      sx={{ m: 0 }}
                      onClick={(e) => e.stopPropagation()}
                    />
                    
                    <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, flex: 1 }}>
                      {getCategoryIcon(group.category)}
                      <Typography variant="subtitle2" fontWeight="medium">
                        {group.name}
                      </Typography>
                      <Chip
                        label={group.category}
                        size="small"
                        color={getCategoryColor(group.category)}
                        sx={{ ml: 'auto' }}
                      />
                    </Box>
                  </Box>
                </AccordionSummary>
                
                <AccordionDetails>
                  <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
                    {group.description}
                  </Typography>
                  
                  <Box sx={{ display: 'flex', flexDirection: 'column', gap: 1 }}>
                    {group.permissions.map((permission) => (
                      <FormControlLabel
                        key={permission.id}
                        control={
                          <Checkbox
                            checked={formData.selectedPermissions.includes(permission.id)}
                            onChange={() => handlePermissionToggle(permission.id)}
                          />
                        }
                        label={
                          <Box>
                            <Typography variant="body2" fontWeight="medium">
                              {permission.name}
                            </Typography>
                            <Typography variant="caption" color="text.secondary">
                              {permission.description}
                            </Typography>
                          </Box>
                        }
                      />
                    ))}
                  </Box>
                </AccordionDetails>
              </Accordion>
            ))}
          </Box>
        </Box>
      </DialogContent>

      <DialogActions sx={{ p: 3 }}>
        <Button onClick={onClose}>
          Cancel
        </Button>
        <Button
          variant="contained"
          startIcon={loading ? <CircularProgress size={16} /> : <Save />}
          onClick={handleSave}
          disabled={!isFormValid() || loading}
        >
          {loading ? 'Saving...' : role ? 'Update Role' : 'Create Role'}
        </Button>
      </DialogActions>
    </Dialog>
  );
};