import React, { useState, useEffect } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  TextField,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Box,
  Typography,
  List,
  ListItem,
  ListItemText,
  ListItemSecondaryAction,
  IconButton,
  Checkbox,
  Divider,
  Accordion,
  AccordionSummary,
  AccordionDetails
} from '@mui/material';
import { Add, Delete, ExpandMore, Assignment } from '@mui/icons-material';
import { OnboardingProcess, OnboardingProcessRequest, OnboardingTask } from '../../types/recruitment';
import { LoaderOverlay } from '../Common/LoaderOverlay';

interface OnboardingDialogProps {
  open: boolean;
  onClose: () => void;
  onSubmit: (data: OnboardingProcessRequest) => Promise<void>;
  candidateName?: string;
  editData?: OnboardingProcess;
}

const defaultTasks: Omit<OnboardingTask, 'id'>[] = [
  {
    title: 'Complete Employment Forms',
    description: 'Fill out all required employment documentation including tax forms, emergency contacts, and bank details',
    category: 'Documentation',
    assignedTo: 'Employee',
    dueDate: '',
    priority: 'High',
    status: 'Pending'
  },
  {
    title: 'IT Equipment Setup',
    description: 'Assign laptop, phone, email account, and necessary software access',
    category: 'IT Setup',
    assignedTo: 'IT',
    dueDate: '',
    priority: 'High',
    status: 'Pending'
  },
  {
    title: 'Office Tour & Introduction',
    description: 'Show office facilities, introduce to team members, and explain office policies',
    category: 'Orientation',
    assignedTo: 'Manager',
    dueDate: '',
    priority: 'Medium',
    status: 'Pending'
  },
  {
    title: 'Company Orientation Training',
    description: 'Complete mandatory company orientation including culture, values, and policies',
    category: 'Training',
    assignedTo: 'HR',
    dueDate: '',
    priority: 'High',
    status: 'Pending'
  },
  {
    title: 'Role-Specific Training',
    description: 'Complete job-specific training and familiarization with tools and processes',
    category: 'Training',
    assignedTo: 'Manager',
    dueDate: '',
    priority: 'Medium',
    status: 'Pending'
  },
  {
    title: 'Benefits Enrollment',
    description: 'Enroll in health insurance, retirement plans, and other benefit programs',
    category: 'Benefits',
    assignedTo: 'HR',
    dueDate: '',
    priority: 'High',
    status: 'Pending'
  }
];

export const OnboardingDialog: React.FC<OnboardingDialogProps> = ({
  open,
  onClose,
  onSubmit,
  candidateName,
  editData
}) => {
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState<OnboardingProcessRequest>({
    employeeId: '',
    candidateId: '',
    startDate: '',
    department: '',
    position: '',
    reportingManager: '',
    buddy: '',
    customTasks: []
  });

  const [selectedTasks, setSelectedTasks] = useState<Set<number>>(new Set());
  const [newTask, setNewTask] = useState<Omit<OnboardingTask, 'id'>>({
    title: '',
    description: '',
    category: 'Documentation',
    assignedTo: 'HR',
    dueDate: '',
    priority: 'Medium',
    status: 'Pending'
  });
  const [showAddTask, setShowAddTask] = useState(false);

  useEffect(() => {
    if (open) {
      if (editData) {
        setFormData({
          employeeId: editData.employeeId,
          candidateId: editData.candidateId,
          startDate: editData.startDate.split('T')[0],
          department: editData.department,
          position: editData.position,
          reportingManager: editData.reportingManager,
          buddy: editData.buddy || '',
          customTasks: editData.tasks
        });
      } else {
        const tomorrow = new Date();
        tomorrow.setDate(tomorrow.getDate() + 1);
        
        setFormData({
          employeeId: `emp-${Date.now()}`,
          candidateId: `candidate-${Date.now()}`,
          startDate: tomorrow.toISOString().split('T')[0],
          department: '',
          position: '',
          reportingManager: '',
          buddy: '',
          customTasks: []
        });
        
        setSelectedTasks(new Set([0, 1, 2, 3, 5]));
      }
      setShowAddTask(false);
    }
  }, [open, editData]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);
    try {
      const selectedDefaultTasks = Array.from(selectedTasks).map(index => ({
        ...defaultTasks[index],
        dueDate: calculateDueDate(defaultTasks[index].assignedTo, formData.startDate)
      }));

      const requestData: OnboardingProcessRequest = {
        ...formData,
        customTasks: [...selectedDefaultTasks, ...(formData.customTasks || [])]
      };

      await onSubmit(requestData);
      onClose();
    } catch (error) {
      console.error('Failed to create onboarding process:', error);
    } finally {
      setLoading(false);
    }
  };

  const calculateDueDate = (assignedTo: string, startDate: string) => {
    const start = new Date(startDate);
    let daysToAdd = 1;
    
    switch (assignedTo) {
      case 'Employee':
        daysToAdd = 1;
        break;
      case 'IT':
        daysToAdd = 2;
        break;
      case 'HR':
        daysToAdd = 3;
        break;
      case 'Manager':
        daysToAdd = 1;
        break;
      default:
        daysToAdd = 7;
    }
    
    const dueDate = new Date(start);
    dueDate.setDate(start.getDate() + daysToAdd);
    return dueDate.toISOString();
  };

  const handleTaskToggle = (index: number) => {
    const newSelected = new Set(selectedTasks);
    if (newSelected.has(index)) {
      newSelected.delete(index);
    } else {
      newSelected.add(index);
    }
    setSelectedTasks(newSelected);
  };

  const addCustomTask = () => {
    if (newTask.title.trim()) {
      setFormData(prev => ({
        ...prev,
        customTasks: [...(prev.customTasks || []), {
          ...newTask,
          dueDate: calculateDueDate(newTask.assignedTo, formData.startDate)
        }]
      }));
      
      setNewTask({
        title: '',
        description: '',
        category: 'Documentation',
        assignedTo: 'HR',
        dueDate: '',
        priority: 'Medium',
        status: 'Pending'
      });
      setShowAddTask(false);
    }
  };

  const removeCustomTask = (index: number) => {
    setFormData(prev => ({
      ...prev,
      customTasks: (prev.customTasks || []).filter((_, i) => i !== index)
    }));
  };

  const getCategoryIcon = (category: string) => {
    switch (category) {
      case 'IT Setup': return '💻';
      case 'Training': return '📚';
      case 'Benefits': return '🏥';
      case 'Legal': return '📋';
      case 'Orientation': return '🏢';
      default: return '📄';
    }
  };

  const getPriorityColor = (priority: string) => {
    switch (priority) {
      case 'Critical': return 'error.main';
      case 'High': return 'warning.main';
      case 'Medium': return 'info.main';
      case 'Low': return 'success.main';
      default: return 'text.secondary';
    }
  };

  return (
    <Dialog open={open} onClose={onClose} maxWidth="md" fullWidth>
      <LoaderOverlay open={loading} />
      <DialogTitle>
        {editData ? 'Update Onboarding Process' : 'Create Onboarding Process'}
        {candidateName && (
          <Typography variant="subtitle2" color="text.secondary">
            For: {candidateName}
          </Typography>
        )}
      </DialogTitle>
      
      <form onSubmit={handleSubmit}>
        <DialogContent sx={{ maxHeight: '70vh', overflowY: 'auto' }}>
          <Box sx={{ display: 'grid', gap: 3, pt: 2 }}>
            <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2 }}>
              <TextField
                label="Employee ID"
                value={formData.employeeId}
                onChange={(e) => setFormData(prev => ({ ...prev, employeeId: e.target.value }))}
                required
              />
              <TextField
                label="Start Date"
                type="date"
                value={formData.startDate}
                onChange={(e) => setFormData(prev => ({ ...prev, startDate: e.target.value }))}
                InputLabelProps={{ shrink: true }}
                required
              />
            </Box>

            <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2 }}>
              <TextField
                label="Department"
                value={formData.department}
                onChange={(e) => setFormData(prev => ({ ...prev, department: e.target.value }))}
                required
              />
              <TextField
                label="Position"
                value={formData.position}
                onChange={(e) => setFormData(prev => ({ ...prev, position: e.target.value }))}
                required
              />
            </Box>

            <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2 }}>
              <TextField
                label="Reporting Manager"
                value={formData.reportingManager}
                onChange={(e) => setFormData(prev => ({ ...prev, reportingManager: e.target.value }))}
                required
              />
              <TextField
                label="Onboarding Buddy (Optional)"
                value={formData.buddy}
                onChange={(e) => setFormData(prev => ({ ...prev, buddy: e.target.value }))}
                placeholder="Assigned team member to help with onboarding"
              />
            </Box>

            <Divider />

            <Box>
              <Typography variant="h6" gutterBottom sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                <Assignment /> Onboarding Tasks
              </Typography>
              <Typography variant="body2" color="text.secondary" gutterBottom>
                Select the tasks that should be included in the onboarding process:
              </Typography>

              <Accordion defaultExpanded>
                <AccordionSummary expandIcon={<ExpandMore />}>
                  <Typography variant="subtitle1">Standard Tasks ({selectedTasks.size} selected)</Typography>
                </AccordionSummary>
                <AccordionDetails>
                  <List>
                    {defaultTasks.map((task, index) => (
                      <ListItem key={index} dense>
                        <Checkbox
                          checked={selectedTasks.has(index)}
                          onChange={() => handleTaskToggle(index)}
                        />
                        <ListItemText
                          primary={
                            <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                              <span>{getCategoryIcon(task.category)}</span>
                              <Typography variant="subtitle2">{task.title}</Typography>
                              <Typography 
                                variant="caption" 
                                sx={{ 
                                  color: getPriorityColor(task.priority),
                                  fontWeight: 'bold',
                                  ml: 'auto'
                                }}
                              >
                                {task.priority}
                              </Typography>
                            </Box>
                          }
                          secondary={
                            <Box>
                              <Typography variant="body2" color="text.secondary">
                                {task.description}
                              </Typography>
                              <Typography variant="caption" color="primary">
                                Assigned to: {task.assignedTo} | Category: {task.category}
                              </Typography>
                            </Box>
                          }
                        />
                      </ListItem>
                    ))}
                  </List>
                </AccordionDetails>
              </Accordion>

              <Accordion>
                <AccordionSummary expandIcon={<ExpandMore />}>
                  <Typography variant="subtitle1">
                    Custom Tasks ({formData.customTasks?.length || 0})
                  </Typography>
                </AccordionSummary>
                <AccordionDetails>
                  {(formData.customTasks?.length || 0) > 0 && (
                    <List>
                      {(formData.customTasks || []).map((task, index) => (
                        <ListItem key={index} dense>
                          <ListItemText
                            primary={task.title}
                            secondary={`${task.description} | ${task.assignedTo} | ${task.category}`}
                          />
                          <ListItemSecondaryAction>
                            <IconButton onClick={() => removeCustomTask(index)} size="small">
                              <Delete />
                            </IconButton>
                          </ListItemSecondaryAction>
                        </ListItem>
                      ))}
                    </List>
                  )}

                  {!showAddTask ? (
                    <Button
                      startIcon={<Add />}
                      onClick={() => setShowAddTask(true)}
                      variant="outlined"
                      sx={{ mt: 1 }}
                    >
                      Add Custom Task
                    </Button>
                  ) : (
                    <Box sx={{ display: 'grid', gap: 2, mt: 2, p: 2, border: 1, borderColor: 'divider', borderRadius: 1 }}>
                      <TextField
                        label="Task Title"
                        value={newTask.title}
                        onChange={(e) => setNewTask(prev => ({ ...prev, title: e.target.value }))}
                        size="small"
                        required
                      />
                      
                      <TextField
                        label="Description"
                        value={newTask.description}
                        onChange={(e) => setNewTask(prev => ({ ...prev, description: e.target.value }))}
                        multiline
                        rows={2}
                        size="small"
                      />

                      <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr 1fr', gap: 1 }}>
                        <FormControl size="small">
                          <InputLabel>Category</InputLabel>
                          <Select
                            value={newTask.category}
                            onChange={(e) => setNewTask(prev => ({ ...prev, category: e.target.value as any }))}
                          >
                            <MenuItem value="Documentation">Documentation</MenuItem>
                            <MenuItem value="IT Setup">IT Setup</MenuItem>
                            <MenuItem value="Training">Training</MenuItem>
                            <MenuItem value="Orientation">Orientation</MenuItem>
                            <MenuItem value="Legal">Legal</MenuItem>
                            <MenuItem value="Benefits">Benefits</MenuItem>
                          </Select>
                        </FormControl>

                        <FormControl size="small">
                          <InputLabel>Assigned To</InputLabel>
                          <Select
                            value={newTask.assignedTo}
                            onChange={(e) => setNewTask(prev => ({ ...prev, assignedTo: e.target.value as any }))}
                          >
                            <MenuItem value="HR">HR</MenuItem>
                            <MenuItem value="Manager">Manager</MenuItem>
                            <MenuItem value="IT">IT</MenuItem>
                            <MenuItem value="Employee">Employee</MenuItem>
                            <MenuItem value="Finance">Finance</MenuItem>
                          </Select>
                        </FormControl>

                        <FormControl size="small">
                          <InputLabel>Priority</InputLabel>
                          <Select
                            value={newTask.priority}
                            onChange={(e) => setNewTask(prev => ({ ...prev, priority: e.target.value as any }))}
                          >
                            <MenuItem value="Low">Low</MenuItem>
                            <MenuItem value="Medium">Medium</MenuItem>
                            <MenuItem value="High">High</MenuItem>
                            <MenuItem value="Critical">Critical</MenuItem>
                          </Select>
                        </FormControl>
                      </Box>

                      <Box sx={{ display: 'flex', gap: 1 }}>
                        <Button onClick={addCustomTask} variant="contained" size="small">
                          Add Task
                        </Button>
                        <Button onClick={() => setShowAddTask(false)} variant="outlined" size="small">
                          Cancel
                        </Button>
                      </Box>
                    </Box>
                  )}
                </AccordionDetails>
              </Accordion>
            </Box>
          </Box>
        </DialogContent>

        <DialogActions>
          <Button onClick={onClose}>Cancel</Button>
          <Button 
            type="submit" 
            variant="contained" 
            disabled={loading || (selectedTasks.size === 0 && (formData.customTasks?.length || 0) === 0)}
          >
            {editData ? 'Update' : 'Create'} Onboarding Process
          </Button>
        </DialogActions>
      </form>
    </Dialog>
  );
};