import React, { useState, useEffect } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  TextField,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Box,
  Chip,
  Typography,
  Autocomplete,
  FormHelperText
} from '@mui/material';
import { DateTimePicker } from '@mui/x-date-pickers/DateTimePicker';
import { Interview, InterviewRequest, Candidate, JobPosting } from '../../types/recruitment';
import { LoaderOverlay } from '../Common/LoaderOverlay';

interface InterviewDialogProps {
  open: boolean;
  onClose: () => void;
  onSubmit: (data: InterviewRequest) => Promise<void>;
  candidate?: Candidate;
  jobPosting?: JobPosting;
  editData?: Interview;
}

const interviewersList = [
  { id: 'int-001', name: 'John Smith', email: 'john.smith@company.com', department: 'Engineering' },
  { id: 'int-002', name: 'Sarah Johnson', email: 'sarah.johnson@company.com', department: 'Product' },
  { id: 'int-003', name: 'Mike Wilson', email: 'mike.wilson@company.com', department: 'HR' },
  { id: 'int-004', name: 'Lisa Chen', email: 'lisa.chen@company.com', department: 'Engineering' },
  { id: 'int-005', name: 'David Brown', email: 'david.brown@company.com', department: 'Sales' }
];

export const InterviewDialog: React.FC<InterviewDialogProps> = ({
  open,
  onClose,
  onSubmit,
  candidate,
  jobPosting,
  editData
}) => {
  const [loading, setLoading] = useState(false);
  const [formData, setFormData] = useState<InterviewRequest>({
    candidateId: '',
    jobId: '',
    type: 'HR',
    round: 1,
    scheduledDate: '',
    duration: 60,
    location: '',
    meetingLink: '',
    interviewerIds: []
  });

  const [selectedInterviewers, setSelectedInterviewers] = useState<typeof interviewersList>([]);
  const [scheduledDateTime, setScheduledDateTime] = useState<Date | null>(null);

  useEffect(() => {
    if (open) {
      if (editData) {
        setFormData({
          candidateId: editData.candidateId,
          jobId: editData.jobId,
          type: editData.type,
          round: editData.round,
          scheduledDate: editData.scheduledDate,
          duration: editData.duration,
          location: editData.location || '',
          meetingLink: editData.meetingLink || '',
          interviewerIds: editData.interviewers.map(i => i.id)
        });
        setScheduledDateTime(new Date(editData.scheduledDate));
        setSelectedInterviewers(
          interviewersList.filter(interviewer => 
            editData.interviewers.some(i => i.id === interviewer.id)
          )
        );
      } else {
        setFormData({
          candidateId: candidate?.id || '',
          jobId: jobPosting?.id || '',
          type: 'HR',
          round: 1,
          scheduledDate: '',
          duration: 60,
          location: '',
          meetingLink: '',
          interviewerIds: []
        });
        setSelectedInterviewers([]);
        setScheduledDateTime(null);
      }
    }
  }, [open, editData, candidate, jobPosting]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    if (!scheduledDateTime) return;

    setLoading(true);
    try {
      const requestData: InterviewRequest = {
        ...formData,
        scheduledDate: scheduledDateTime.toISOString(),
        interviewerIds: selectedInterviewers.map(i => i.id)
      };
      await onSubmit(requestData);
      onClose();
    } catch (error) {
      console.error('Failed to schedule interview:', error);
    } finally {
      setLoading(false);
    }
  };

  const getInterviewTypeDescription = (type: string) => {
    switch (type) {
      case 'Phone': return 'Initial phone screening with HR';
      case 'Video': return 'Video call interview';
      case 'In-Person': return 'Face-to-face interview at office';
      case 'Technical': return 'Technical assessment and coding';
      case 'HR': return 'HR round for culture fit';
      case 'Final': return 'Final round with senior management';
      default: return '';
    }
  };

  return (
    <Dialog open={open} onClose={onClose} maxWidth="md" fullWidth>
      <LoaderOverlay open={loading} />
      <DialogTitle>
        {editData ? 'Reschedule Interview' : 'Schedule Interview'}
        {candidate && (
          <Typography variant="subtitle2" color="text.secondary">
            For: {candidate.firstName} {candidate.lastName}
          </Typography>
        )}
      </DialogTitle>
      
      <form onSubmit={handleSubmit}>
        <DialogContent>
          <Box sx={{ display: 'grid', gap: 3, pt: 2 }}>
            {candidate && (
              <Box sx={{ p: 2, bgcolor: 'grey.50', borderRadius: 1 }}>
                <Typography variant="subtitle2" gutterBottom>Candidate Information</Typography>
                <Typography variant="body2">
                  <strong>{candidate.firstName} {candidate.lastName}</strong><br />
                  {candidate.email} | {candidate.phone}<br />
                  {candidate.currentPosition} at {candidate.currentCompany}<br />
                  Experience: {candidate.totalExperience} years
                </Typography>
              </Box>
            )}

            <Box sx={{ display: 'grid', gridTemplateColumns: '1fr 1fr', gap: 2 }}>
              <FormControl required>
                <InputLabel>Interview Type</InputLabel>
                <Select
                  value={formData.type}
                  onChange={(e) => setFormData(prev => ({ ...prev, type: e.target.value as any }))}
                >
                  <MenuItem value="HR">HR Round</MenuItem>
                  <MenuItem value="Phone">Phone Screening</MenuItem>
                  <MenuItem value="Video">Video Interview</MenuItem>
                  <MenuItem value="Technical">Technical Round</MenuItem>
                  <MenuItem value="In-Person">In-Person</MenuItem>
                  <MenuItem value="Final">Final Round</MenuItem>
                </Select>
                <FormHelperText>{getInterviewTypeDescription(formData.type)}</FormHelperText>
              </FormControl>

              <TextField
                label="Round Number"
                type="number"
                value={formData.round}
                onChange={(e) => setFormData(prev => ({ ...prev, round: parseInt(e.target.value) || 1 }))}
                required
                inputProps={{ min: 1, max: 5 }}
              />
            </Box>

            <DateTimePicker
              label="Interview Date & Time"
              value={scheduledDateTime}
              onChange={(newValue) => setScheduledDateTime(newValue)}
              minDateTime={new Date()}
              slotProps={{
                textField: {
                  required: true,
                  fullWidth: true,
                  helperText: 'Select date and time for the interview'
                }
              }}
            />

            <TextField
              label="Duration (minutes)"
              type="number"
              value={formData.duration}
              onChange={(e) => setFormData(prev => ({ ...prev, duration: parseInt(e.target.value) || 60 }))}
              required
              inputProps={{ min: 15, max: 240, step: 15 }}
              helperText="Typical durations: 30 min (phone), 60 min (technical), 90 min (final)"
            />

            <Autocomplete
              multiple
              options={interviewersList}
              getOptionLabel={(option) => `${option.name} (${option.department})`}
              value={selectedInterviewers}
              onChange={(_, newValue) => setSelectedInterviewers(newValue)}
              renderInput={(params) => (
                <TextField
                  {...params}
                  label="Interviewers"
                  placeholder="Select interviewers"
                  required={selectedInterviewers.length === 0}
                  helperText="Select one or more interviewers for this round"
                />
              )}
              renderTags={(value, getTagProps) =>
                value.map((option, index) => {
                  const { key, ...tagProps } = getTagProps({ index });
                  return (
                    <Chip
                      key={option.id}
                      label={option.name}
                      size="small"
                      {...tagProps}
                    />
                  );
                })
              }
            />

            {(formData.type === 'Video' || formData.type === 'Phone') && (
              <TextField
                label="Meeting Link"
                value={formData.meetingLink}
                onChange={(e) => setFormData(prev => ({ ...prev, meetingLink: e.target.value }))}
                placeholder="https://meet.google.com/xxx-xxx-xxx"
                helperText="Video call link (Google Meet, Zoom, Teams, etc.)"
              />
            )}

            {formData.type === 'In-Person' && (
              <TextField
                label="Location"
                value={formData.location}
                onChange={(e) => setFormData(prev => ({ ...prev, location: e.target.value }))}
                required
                placeholder="Conference Room A, Floor 3"
                helperText="Specify the meeting room or location"
              />
            )}

            <Box sx={{ p: 2, bgcolor: 'info.light', borderRadius: 1, color: 'info.contrastText' }}>
              <Typography variant="subtitle2" gutterBottom>Interview Details Summary</Typography>
              <Typography variant="body2">
                <strong>Type:</strong> {formData.type} - Round {formData.round}<br />
                <strong>Duration:</strong> {formData.duration} minutes<br />
                <strong>Interviewers:</strong> {selectedInterviewers.length > 0 ? selectedInterviewers.map(i => i.name).join(', ') : 'None selected'}<br />
                {formData.type === 'In-Person' && formData.location && (
                  <>
                    <strong>Location:</strong> {formData.location}<br />
                  </>
                )}
                {(formData.type === 'Video' || formData.type === 'Phone') && formData.meetingLink && (
                  <>
                    <strong>Meeting Link:</strong> {formData.meetingLink}<br />
                  </>
                )}
              </Typography>
            </Box>
          </Box>
        </DialogContent>

        <DialogActions>
          <Button onClick={onClose}>Cancel</Button>
          <Button 
            type="submit" 
            variant="contained" 
            disabled={loading || !scheduledDateTime || selectedInterviewers.length === 0}
          >
            {editData ? 'Reschedule' : 'Schedule'} Interview
          </Button>
        </DialogActions>
      </form>
    </Dialog>
  );
};