import React, { useState, useEffect } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  TextField,
  Button,
  Typography,
  Box,
  IconButton,
  InputAdornment,
  Alert,
  Tabs,
  Tab,
  CircularProgress,
  Switch,
  FormControlLabel,
} from '@mui/material';
import {
  Visibility,
  VisibilityOff,
  Close,
  Science,
} from '@mui/icons-material';
import { MailSettings } from '../../types/services/mailService.types';

interface SettingsDialogProps {
  open: boolean;
  onClose: () => void;
  onSave: (settings: any) => void;
  editingSettings?: MailSettings | null;
  loading?: boolean;
  onTestConnection?: (smtp: any, imap: any) => void;
  testingConnection?: boolean;
}

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

function TabPanel(props: TabPanelProps) {
  const { children, value, index, ...other } = props;
  return (
    <div
      role="tabpanel"
      hidden={value !== index}
      id={`settings-tabpanel-${index}`}
      {...other}
    >
      {value === index && <Box sx={{ pt: 2 }}>{children}</Box>}
    </div>
  );
}

export const SettingsDialog: React.FC<SettingsDialogProps> = ({
  open,
  onClose,
  onSave,
  editingSettings,
  loading = false,
  onTestConnection,
  testingConnection = false,
}) => {
  const [tabValue, setTabValue] = useState(0);
  const [showSmtpPassword, setShowSmtpPassword] = useState(false);
  const [showImapPassword, setShowImapPassword] = useState(false);
  const [formData, setFormData] = useState({
    name: '',
    smtpHost: '',
    smtpPort: '587',
    smtpSecure: false,
    smtpUser: '',
    smtpPass: '',
    imapHost: '',
    imapPort: '993',
    imapSecure: true,
    imapUser: '',
    imapPass: '',
    defaultFrom: '',
    defaultFromName: '',
    isDefault: false,
    isActive: true,
  });

  useEffect(() => {
    if (editingSettings) {
      setFormData({
        name: editingSettings.name,
        smtpHost: editingSettings.smtp.host,
        smtpPort: editingSettings.smtp.port.toString(),
        smtpSecure: editingSettings.smtp.secure,
        smtpUser: editingSettings.smtp.auth.user,
        smtpPass: editingSettings.smtp.auth.pass,
        imapHost: editingSettings.imap.host,
        imapPort: editingSettings.imap.port.toString(),
        imapSecure: editingSettings.imap.secure,
        imapUser: editingSettings.imap.auth.user,
        imapPass: editingSettings.imap.auth.pass,
        defaultFrom: editingSettings.defaultFrom,
        defaultFromName: editingSettings.defaultFromName,
        isDefault: editingSettings.isDefault,
        isActive: editingSettings.isActive,
      });
    } else {
      // Reset form for new settings
      setFormData({
        name: '',
        smtpHost: '',
        smtpPort: '587',
        smtpSecure: false,
        smtpUser: '',
        smtpPass: '',
        imapHost: '',
        imapPort: '993',
        imapSecure: true,
        imapUser: '',
        imapPass: '',
        defaultFrom: '',
        defaultFromName: '',
        isDefault: false,
        isActive: true,
      });
    }
  }, [editingSettings, open]);

  const handleInputChange = (field: string, value: any) => {
    setFormData({ ...formData, [field]: value });
  };

  const handleTestConnection = () => {
    if (onTestConnection) {
      const smtpConfig = {
        host: formData.smtpHost,
        port: parseInt(formData.smtpPort),
        secure: formData.smtpSecure,
        auth: {
          user: formData.smtpUser,
          pass: formData.smtpPass,
        },
      };

      const imapConfig = {
        host: formData.imapHost,
        port: parseInt(formData.imapPort),
        secure: formData.imapSecure,
        auth: {
          user: formData.imapUser,
          pass: formData.imapPass,
        },
      };

      onTestConnection(smtpConfig, imapConfig);
    }
  };

  const handleSave = () => {
    onSave(formData);
  };

  const isFormValid = () => {
    return (
      formData.name &&
      formData.smtpHost &&
      formData.smtpPort &&
      formData.smtpUser &&
      formData.smtpPass &&
      formData.imapHost &&
      formData.imapPort &&
      formData.imapUser &&
      formData.imapPass &&
      formData.defaultFrom &&
      formData.defaultFromName
    );
  };

  const GeneralTab = () => (
    <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
      <TextField
        label="Configuration Name"
        value={formData.name}
        onChange={(e) => handleInputChange('name', e.target.value)}
        fullWidth
        required
      />
      <TextField
        label="Default From Email"
        type="email"
        value={formData.defaultFrom}
        onChange={(e) => handleInputChange('defaultFrom', e.target.value)}
        fullWidth
        required
      />
      <TextField
        label="Default From Name"
        value={formData.defaultFromName}
        onChange={(e) => handleInputChange('defaultFromName', e.target.value)}
        fullWidth
        required
      />
      <FormControlLabel
        control={
          <Switch
            checked={formData.isDefault}
            onChange={(e) => handleInputChange('isDefault', e.target.checked)}
          />
        }
        label="Set as Default Configuration"
      />
      <FormControlLabel
        control={
          <Switch
            checked={formData.isActive}
            onChange={(e) => handleInputChange('isActive', e.target.checked)}
          />
        }
        label="Active"
      />
    </Box>
  );

  const SMTPTab = () => (
    <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
      <Alert severity="info" sx={{ mb: 2 }}>
        SMTP configuration for sending emails. Common providers:
        <br />• Gmail: smtp.gmail.com, port 587 (TLS) or 465 (SSL)
        <br />• Outlook: smtp-mail.outlook.com, port 587 (TLS)
      </Alert>
      <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
        <Box sx={{ flex: '2 1 200px' }}>
          <TextField
            label="SMTP Host"
            value={formData.smtpHost}
            onChange={(e) => handleInputChange('smtpHost', e.target.value)}
            fullWidth
            required
            placeholder="e.g., smtp.gmail.com"
          />
        </Box>
        <Box sx={{ flex: '1 1 100px' }}>
          <TextField
            label="Port"
            type="number"
            value={formData.smtpPort}
            onChange={(e) => handleInputChange('smtpPort', e.target.value)}
            fullWidth
            required
          />
        </Box>
      </Box>
      <FormControlLabel
        control={
          <Switch
            checked={formData.smtpSecure}
            onChange={(e) => handleInputChange('smtpSecure', e.target.checked)}
          />
        }
        label="Use SSL/TLS (recommended for port 465)"
      />
      <TextField
        label="Username/Email"
        value={formData.smtpUser}
        onChange={(e) => handleInputChange('smtpUser', e.target.value)}
        fullWidth
        required
      />
      <TextField
        label="Password"
        type={showSmtpPassword ? 'text' : 'password'}
        value={formData.smtpPass}
        onChange={(e) => handleInputChange('smtpPass', e.target.value)}
        fullWidth
        required
        InputProps={{
          endAdornment: (
            <InputAdornment position="end">
              <IconButton
                onClick={() => setShowSmtpPassword(!showSmtpPassword)}
                edge="end"
              >
                {showSmtpPassword ? <VisibilityOff /> : <Visibility />}
              </IconButton>
            </InputAdornment>
          ),
        }}
      />
    </Box>
  );

  const IMAPTab = () => (
    <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
      <Alert severity="info" sx={{ mb: 2 }}>
        IMAP configuration for receiving emails. Common providers:
        <br />• Gmail: imap.gmail.com, port 993 (SSL)
        <br />• Outlook: imap-mail.outlook.com, port 993 (SSL)
      </Alert>
      <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap' }}>
        <Box sx={{ flex: '2 1 200px' }}>
          <TextField
            label="IMAP Host"
            value={formData.imapHost}
            onChange={(e) => handleInputChange('imapHost', e.target.value)}
            fullWidth
            required
            placeholder="e.g., imap.gmail.com"
          />
        </Box>
        <Box sx={{ flex: '1 1 100px' }}>
          <TextField
            label="Port"
            type="number"
            value={formData.imapPort}
            onChange={(e) => handleInputChange('imapPort', e.target.value)}
            fullWidth
            required
          />
        </Box>
      </Box>
      <FormControlLabel
        control={
          <Switch
            checked={formData.imapSecure}
            onChange={(e) => handleInputChange('imapSecure', e.target.checked)}
          />
        }
        label="Use SSL/TLS (recommended)"
      />
      <TextField
        label="Username/Email"
        value={formData.imapUser}
        onChange={(e) => handleInputChange('imapUser', e.target.value)}
        fullWidth
        required
      />
      <TextField
        label="Password"
        type={showImapPassword ? 'text' : 'password'}
        value={formData.imapPass}
        onChange={(e) => handleInputChange('imapPass', e.target.value)}
        fullWidth
        required
        InputProps={{
          endAdornment: (
            <InputAdornment position="end">
              <IconButton
                onClick={() => setShowImapPassword(!showImapPassword)}
                edge="end"
              >
                {showImapPassword ? <VisibilityOff /> : <Visibility />}
              </IconButton>
            </InputAdornment>
          ),
        }}
      />
    </Box>
  );

  return (
    <Dialog
      open={open}
      onClose={onClose}
      maxWidth="md"
      fullWidth
      PaperProps={{
        sx: { minHeight: '70vh' }
      }}
    >
      <DialogTitle sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
        <Typography variant="h6">
          {editingSettings ? 'Edit Mail Settings' : 'Add Mail Settings'}
        </Typography>
        <IconButton onClick={onClose}>
          <Close />
        </IconButton>
      </DialogTitle>

      <DialogContent>
        <Tabs value={tabValue} onChange={(_, newValue) => setTabValue(newValue)}>
          <Tab label="General" />
          <Tab label="SMTP (Send)" />
          <Tab label="IMAP (Receive)" />
        </Tabs>

        <TabPanel value={tabValue} index={0}>
          <GeneralTab />
        </TabPanel>
        <TabPanel value={tabValue} index={1}>
          <SMTPTab />
        </TabPanel>
        <TabPanel value={tabValue} index={2}>
          <IMAPTab />
        </TabPanel>
      </DialogContent>

      <DialogActions sx={{ p: 3 }}>
        <Button onClick={onClose}>Cancel</Button>
        <Button
          variant="outlined"
          startIcon={testingConnection ? <CircularProgress size={16} /> : <Science />}
          onClick={handleTestConnection}
          disabled={testingConnection || !formData.smtpHost || !formData.imapHost}
        >
          Test Connection
        </Button>
        <Button
          variant="contained"
          onClick={handleSave}
          disabled={!isFormValid() || loading}
        >
          {loading ? <CircularProgress size={20} /> : 'Save'}
        </Button>
      </DialogActions>
    </Dialog>
  );
};