import React from 'react';
import { Link, useLocation } from 'react-router-dom';
import {
  Drawer,
  List,
  ListItem,
  ListItemButton,
  ListItemIcon,
  ListItemText,
  Toolbar,
  Typography,
  Box,
  useTheme,
  useMediaQuery,
  Divider,
  Avatar,
} from '@mui/material';
import {
  Dashboard,
  People,
  Business,
  Schedule,
  EventNote,
  Payment,
  Assessment,
  Email,
  CalendarToday,
  Security,
  Logout,
  AccountBalance,
  Work,
} from '@mui/icons-material';
import { useAuth } from '../../contexts/AuthContext';

const drawerWidth = 280;

const menuItems = [
  { path: '/dashboard', label: 'Dashboard', icon: Dashboard },
  { path: '/employees', label: 'Employees', icon: People },
  { path: '/departments', label: 'Departments', icon: Business },
  { path: '/attendance', label: 'Attendance', icon: Schedule },
  { path: '/leaves', label: 'Leave Management', icon: EventNote },
  { path: '/holidays', label: 'Holidays', icon: CalendarToday },
  { path: '/payroll', label: 'Payroll', icon: Payment },
  { path: '/recruitment', label: 'Recruitment', icon: Work },
  { path: '/compliance', label: 'Compliance & Policies', icon: AccountBalance },
  { path: '/reports', label: 'Reports', icon: Assessment },
  { path: '/user-management', label: 'User Management', icon: Security },
  { path: '/mail-settings', label: 'Mail Settings', icon: Email },
];

interface SidebarProps {
  mobileOpen: boolean;
  handleDrawerToggle: () => void;
}

export const Sidebar: React.FC<SidebarProps> = ({ mobileOpen, handleDrawerToggle }) => {
  const location = useLocation();
  const { logout, user } = useAuth();
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('sm'));

  const drawer = (
    <Box sx={{ height: '100%', display: 'flex', flexDirection: 'column' }}>
      <Toolbar sx={{ 
        minHeight: { xs: '56px', sm: '64px' },
        px: { xs: 2, sm: 3 },
        justifyContent: 'center'
      }}>
        <Typography 
          variant={isMobile ? "h6" : "h5"} 
          noWrap 
          component="div"
          sx={{ 
            fontWeight: 600,
            color: 'primary.main',
            textAlign: 'center'
          }}
        >
          HR Portal
        </Typography>
      </Toolbar>
      
      <Divider />
      
      {/* User Info Section for Mobile */}
      {isMobile && user && (
        <Box sx={{ p: 2, textAlign: 'center', bgcolor: 'grey.50' }}>
          <Avatar sx={{ width: 48, height: 48, mx: 'auto', mb: 1 }}>
            {user.name.charAt(0)}
          </Avatar>
          <Typography variant="body2" fontWeight={500}>
            {user.name}
          </Typography>
          <Typography variant="caption" color="textSecondary">
            {user.role.toUpperCase()}
          </Typography>
        </Box>
      )}
      
      <Divider />
      
      <List sx={{ 
        flex: 1,
        py: 1,
        px: { xs: 1, sm: 2 }
      }}>
        {menuItems.map((item) => {
          const Icon = item.icon;
          const isActive = location.pathname === item.path;
          
          return (
            <ListItem key={item.path} disablePadding sx={{ mb: 0.5 }}>
              <ListItemButton
                component={Link}
                to={item.path}
                onClick={isMobile ? handleDrawerToggle : undefined}
                selected={isActive}
                sx={{
                  borderRadius: 2,
                  mx: 1,
                  minHeight: { xs: 48, sm: 52 },
                  backgroundColor: isActive ? 'primary.main' : 'transparent',
                  color: isActive ? 'primary.contrastText' : 'inherit',
                  '&:hover': {
                    backgroundColor: isActive ? 'primary.dark' : 'action.hover',
                  },
                  '&.Mui-selected': {
                    backgroundColor: 'primary.main',
                    color: 'primary.contrastText',
                    '&:hover': {
                      backgroundColor: 'primary.dark',
                    },
                  },
                }}
              >
                <ListItemIcon sx={{ 
                  color: 'inherit',
                  minWidth: { xs: 40, sm: 56 }
                }}>
                  <Icon fontSize={isMobile ? 'medium' : 'small'} />
                </ListItemIcon>
                <ListItemText 
                  primary={item.label}
                  primaryTypographyProps={{
                    fontSize: { xs: '0.875rem', sm: '1rem' },
                    fontWeight: isActive ? 600 : 400,
                  }}
                />
              </ListItemButton>
            </ListItem>
          );
        })}
      </List>
      
      <Divider />
      
      <Box sx={{ p: 1 }}>
        <ListItem disablePadding>
          <ListItemButton 
            onClick={logout}
            sx={{
              borderRadius: 2,
              mx: 1,
              minHeight: { xs: 48, sm: 52 },
              color: 'error.main',
              '&:hover': {
                backgroundColor: 'error.light',
                color: 'error.contrastText',
              },
            }}
          >
            <ListItemIcon sx={{ 
              color: 'inherit',
              minWidth: { xs: 40, sm: 56 }
            }}>
              <Logout fontSize={isMobile ? 'medium' : 'small'} />
            </ListItemIcon>
            <ListItemText 
              primary="Logout"
              primaryTypographyProps={{
                fontSize: { xs: '0.875rem', sm: '1rem' },
                fontWeight: 500,
              }}
            />
          </ListItemButton>
        </ListItem>
      </Box>
    </Box>
  );

  return (
    <Box
      component="nav"
      sx={{ width: { sm: drawerWidth }, flexShrink: { sm: 0 } }}
    >
      <Drawer
        variant="temporary"
        open={mobileOpen}
        onClose={handleDrawerToggle}
        ModalProps={{
          keepMounted: true,
        }}
        sx={{
          display: { xs: 'block', sm: 'none' },
          '& .MuiDrawer-paper': { 
            boxSizing: 'border-box', 
            width: { xs: '85vw', sm: drawerWidth },
            maxWidth: drawerWidth,
          },
        }}
      >
        {drawer}
      </Drawer>
      <Drawer
        variant="permanent"
        sx={{
          display: { xs: 'none', sm: 'block' },
          '& .MuiDrawer-paper': { boxSizing: 'border-box', width: drawerWidth },
        }}
        open
      >
        {drawer}
      </Drawer>
    </Box>
  );
};