import React, { useState, useEffect } from 'react';
import {
  Box,
  Card,
  CardContent,
  Typography,
  IconButton,
  Chip,
  Tooltip,
  Badge,
  CircularProgress,
  Alert,
  useTheme,
} from '@mui/material';
import {
  ChevronLeft,
  ChevronRight,
  Event,
  Today,
} from '@mui/icons-material';
import { format, 
  startOfMonth, 
  endOfMonth, 
  startOfWeek, 
  endOfWeek, 
  addDays, 
  isSameMonth, 
  isSameDay, 
  addMonths, 
  subMonths,
  isToday,
  isWeekend
} from 'date-fns';
import { Holiday, HolidayCalendarView } from '../../types';
import { holidayService } from '../../services';

interface HolidayCalendarProps {
  organizationId?: string;
  onDateClick?: (date: Date, holidays: Holiday[]) => void;
  onHolidayClick?: (holiday: Holiday) => void;
  showAddButton?: boolean;
  height?: number | string;
}

interface CalendarDay {
  date: Date;
  holidays: Holiday[];
  isCurrentMonth: boolean;
  isToday: boolean;
  isWeekend: boolean;
}

const HOLIDAY_COLORS: Record<string, string> = {
  public: '#f44336',
  company: '#2196f3',
  optional: '#ff9800',
  religious: '#9c27b0',
};

export const HolidayCalendar: React.FC<HolidayCalendarProps> = ({
  organizationId,
  onDateClick,
  onHolidayClick,
  showAddButton = false,
  height = 600,
}) => {
  const theme = useTheme();
  const [currentDate, setCurrentDate] = useState(new Date());
  const [calendarView, setCalendarView] = useState<HolidayCalendarView | null>(null);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');

  useEffect(() => {
    loadCalendarData();
  }, [currentDate, organizationId]);

  const loadCalendarData = async () => {
    setLoading(true);
    setError('');
    
    try {
      const year = currentDate.getFullYear();
      const month = currentDate.getMonth() + 1;
      
      const view = await holidayService.getHolidayCalendarView(year, month, organizationId);
      setCalendarView(view);
    } catch (err) {
      setError('Failed to load calendar data');
      console.error('Calendar load error:', err);
    } finally {
      setLoading(false);
    }
  };

  const generateCalendarDays = (): CalendarDay[] => {
    if (!calendarView) return [];

    const monthStart = startOfMonth(currentDate);
    const monthEnd = endOfMonth(currentDate);
    const startDate = startOfWeek(monthStart);
    const endDate = endOfWeek(monthEnd);

    const days: CalendarDay[] = [];
    let day = startDate;

    while (day <= endDate) {
      const dayString = format(day, 'yyyy-MM-dd');
      const dayHolidays = calendarView.holidays.filter(
        holiday => holiday.date === dayString
      );

      days.push({
        date: new Date(day),
        holidays: dayHolidays,
        isCurrentMonth: isSameMonth(day, currentDate),
        isToday: isToday(day),
        isWeekend: isWeekend(day),
      });

      day = addDays(day, 1);
    }

    return days;
  };

  const handlePrevMonth = () => {
    setCurrentDate(subMonths(currentDate, 1));
  };

  const handleNextMonth = () => {
    setCurrentDate(addMonths(currentDate, 1));
  };

  const handleToday = () => {
    setCurrentDate(new Date());
  };

  const handleDateClick = (day: CalendarDay) => {
    if (onDateClick) {
      onDateClick(day.date, day.holidays);
    }
  };

  const handleHolidayClick = (holiday: Holiday, event: React.MouseEvent) => {
    event.stopPropagation();
    if (onHolidayClick) {
      onHolidayClick(holiday);
    }
  };

  const getHolidayColor = (holiday: Holiday): string => {
    return holiday.color || HOLIDAY_COLORS[holiday.type] || theme.palette.primary.main;
  };

  const renderCalendarDay = (day: CalendarDay, index: number) => {
    const hasHolidays = day.holidays.length > 0;
    const dayNumber = format(day.date, 'd');

    return (
      <Box
        key={index}
        sx={{
          minHeight: 80,
          border: '1px solid',
          borderColor: theme.palette.divider,
          backgroundColor: day.isCurrentMonth 
            ? (day.isToday ? theme.palette.action.selected : 'background.paper')
            : 'action.hover',
          cursor: 'pointer',
          p: 1,
          display: 'flex',
          flexDirection: 'column',
          '&:hover': {
            backgroundColor: day.isCurrentMonth 
              ? theme.palette.action.hover 
              : theme.palette.action.selected,
          },
          opacity: day.isCurrentMonth ? 1 : 0.6,
        }}
        onClick={() => handleDateClick(day)}
      >
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 1 }}>
          <Typography
            variant="body2"
            sx={{
              fontWeight: day.isToday ? 'bold' : 'normal',
              color: day.isToday 
                ? theme.palette.primary.main 
                : day.isWeekend 
                  ? theme.palette.text.secondary 
                  : 'text.primary',
            }}
          >
            {dayNumber}
          </Typography>
          {hasHolidays && (
            <Badge badgeContent={day.holidays.length} color="primary">
              <Event fontSize="small" />
            </Badge>
          )}
        </Box>

        <Box sx={{ flex: 1, display: 'flex', flexDirection: 'column', gap: 0.5 }}>
          {day.holidays.slice(0, 2).map((holiday) => (
            <Tooltip key={holiday.id} title={holiday.description || holiday.name}>
              <Chip
                label={holiday.name}
                size="small"
                sx={{
                  fontSize: '0.65rem',
                  height: 'auto',
                  minHeight: 20,
                  backgroundColor: getHolidayColor(holiday),
                  color: 'white',
                  '& .MuiChip-label': {
                    px: 1,
                    py: 0.25,
                    whiteSpace: 'nowrap',
                    overflow: 'hidden',
                    textOverflow: 'ellipsis',
                    maxWidth: '100%',
                  },
                }}
                onClick={(e) => handleHolidayClick(holiday, e)}
              />
            </Tooltip>
          ))}
          
          {day.holidays.length > 2 && (
            <Chip
              label={`+${day.holidays.length - 2} more`}
              size="small"
              variant="outlined"
              sx={{
                fontSize: '0.65rem',
                height: 'auto',
                minHeight: 20,
                '& .MuiChip-label': {
                  px: 1,
                  py: 0.25,
                },
              }}
            />
          )}
        </Box>
      </Box>
    );
  };

  const calendarDays = generateCalendarDays();
  const weekDays = ['Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat'];

  return (
    <Card sx={{ height }}>
      <CardContent sx={{ height: '100%', display: 'flex', flexDirection: 'column', p: 2 }}>
        {/* Calendar Header */}
        <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 2 }}>
          <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
            <IconButton onClick={handlePrevMonth} disabled={loading}>
              <ChevronLeft />
            </IconButton>
            <Typography variant="h6" sx={{ minWidth: 200, textAlign: 'center' }}>
              {format(currentDate, 'MMMM yyyy')}
            </Typography>
            <IconButton onClick={handleNextMonth} disabled={loading}>
              <ChevronRight />
            </IconButton>
          </Box>
          
          <Box sx={{ display: 'flex', gap: 1 }}>
            <IconButton onClick={handleToday} disabled={loading} title="Today">
              <Today />
            </IconButton>
          </Box>
        </Box>

        {error && (
          <Alert severity="error" sx={{ mb: 2 }}>
            {error}
          </Alert>
        )}

        {loading ? (
          <Box sx={{ display: 'flex', justifyContent: 'center', alignItems: 'center', flex: 1 }}>
            <CircularProgress />
          </Box>
        ) : (
          <Box sx={{ flex: 1, display: 'flex', flexDirection: 'column' }}>
            {/* Week Days Header */}
            <Box sx={{ display: 'grid', gridTemplateColumns: 'repeat(7, 1fr)', mb: 1 }}>
              {weekDays.map((day) => (
                <Box key={day} sx={{ textAlign: 'center', py: 1 }}>
                  <Typography variant="subtitle2" color="text.secondary" fontWeight="medium">
                    {day}
                  </Typography>
                </Box>
              ))}
            </Box>

            {/* Calendar Grid */}
            <Box sx={{ flex: 1, display: 'grid', gridTemplateColumns: 'repeat(7, 1fr)', gap: 0 }}>
              {calendarDays.map((day, index) => renderCalendarDay(day, index))}
            </Box>

            {/* Legend */}
            <Box sx={{ mt: 2, display: 'flex', gap: 1, flexWrap: 'wrap', justifyContent: 'center' }}>
              {Object.entries(HOLIDAY_COLORS).map(([type, color]) => (
                <Box key={type} sx={{ display: 'flex', alignItems: 'center', gap: 0.5 }}>
                  <Box 
                    sx={{ 
                      width: 12, 
                      height: 12, 
                      backgroundColor: color, 
                      borderRadius: '50%' 
                    }} 
                  />
                  <Typography variant="caption" sx={{ textTransform: 'capitalize' }}>
                    {type}
                  </Typography>
                </Box>
              ))}
            </Box>
          </Box>
        )}
      </CardContent>
    </Card>
  );
};