import React, { useState, useEffect } from 'react';
import { LoaderOverlay } from '../Common';
import {
  Box,
  Typography,
  Button,
  Card,
  CardContent,
  CardActions,
  IconButton,
  Menu,
  MenuItem,
  ListItemIcon,
  ListItemText,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogContentText,
  DialogActions,
  Chip,
  Avatar,
  Grid,
  TextField,
  InputAdornment,
  FormControl,
  InputLabel,
  Select,
  CircularProgress,
  Alert,
  Tooltip,
  Fab,
} from '@mui/material';
import {
  Add,
  MoreVert,
  Edit,
  Delete,
  Visibility,
  Download,
  Search,
  FilterList,
  Policy,
  CheckCircle,
  Schedule,
  Warning,
  Assignment,
} from '@mui/icons-material';
import { HRPolicy, PolicyFilter, ComplianceStats } from '../../types';
import { complianceService } from '../../services';
import { PolicyDialog } from './PolicyDialog';

export const PolicyManagement: React.FC = () => {
  const [policies, setPolicies] = useState<HRPolicy[]>([]);
  const [stats, setStats] = useState<ComplianceStats | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [filter, setFilter] = useState<PolicyFilter>({});
  const [selectedPolicy, setSelectedPolicy] = useState<HRPolicy | null>(null);
  const [dialogType, setDialogType] = useState<'create' | 'edit' | 'view' | null>(null);
  const [deleteConfirmOpen, setDeleteConfirmOpen] = useState(false);
  const [policyToDelete, setPolicyToDelete] = useState<HRPolicy | null>(null);
  const [anchorEl, setAnchorEl] = useState<null | HTMLElement>(null);
  const [operationLoading, setOperationLoading] = useState<string | null>(null);

  useEffect(() => {
    loadData();
  }, [filter]);

  const loadData = async () => {
    try {
      setLoading(true);
      setOperationLoading('Loading policies...');
      const [policiesResponse, statsResponse] = await Promise.all([
        complianceService.getPolicies(filter),
        complianceService.getComplianceStats()
      ]);
      
      setPolicies(policiesResponse.data);
      setStats(statsResponse);
      setError('');
    } catch (err: any) {
      setError(err.message || 'Failed to load policies');
    } finally {
      setLoading(false);
      setOperationLoading(null);
    }
  };

  const handleMenuClick = (event: React.MouseEvent<HTMLElement>, policy: HRPolicy) => {
    setAnchorEl(event.currentTarget);
    setSelectedPolicy(policy);
  };

  const handleMenuClose = () => {
    setAnchorEl(null);
    setSelectedPolicy(null);
  };

  const handleCreatePolicy = () => {
    setSelectedPolicy(null);
    setDialogType('create');
  };

  const handleEditPolicy = () => {
    setDialogType('edit');
    handleMenuClose();
  };

  const handleViewPolicy = () => {
    setDialogType('view');
    handleMenuClose();
  };

  const handleDeletePolicy = () => {
    setPolicyToDelete(selectedPolicy);
    setDeleteConfirmOpen(true);
    handleMenuClose();
  };

  const confirmDelete = async () => {
    if (!policyToDelete) return;

    try {
      setOperationLoading('Deleting policy...');
      await complianceService.deletePolicy(policyToDelete.id);
      setPolicies(policies.filter(policy => policy.id !== policyToDelete.id));
      setDeleteConfirmOpen(false);
      setPolicyToDelete(null);
    } catch (err: any) {
      setError(err.message || 'Failed to delete policy');
    } finally {
      setOperationLoading(null);
    }
  };

  const handleDialogSave = (policy: HRPolicy) => {
    if (dialogType === 'create') {
      setPolicies([policy, ...policies]);
    } else if (dialogType === 'edit') {
      setPolicies(policies.map(p => p.id === policy.id ? policy : p));
    }
    setDialogType(null);
    setSelectedPolicy(null);
  };

  const handleDialogClose = () => {
    setDialogType(null);
    setSelectedPolicy(null);
  };

  const getStatusColor = (status: HRPolicy['status']) => {
    switch (status) {
      case 'active': return 'success';
      case 'draft': return 'warning';
      case 'expired': return 'error';
      case 'archived': return 'default';
      default: return 'default';
    }
  };

  const getStatusIcon = (status: HRPolicy['status']) => {
    switch (status) {
      case 'active': return <CheckCircle />;
      case 'draft': return <Schedule />;
      case 'expired': return <Warning />;
      case 'archived': return <Assignment />;
      default: return <Policy />;
    }
  };

  const getCategoryColor = (category: HRPolicy['category']) => {
    const colors: Record<HRPolicy['category'], 'primary' | 'secondary' | 'success' | 'warning' | 'error' | 'info'> = {
      attendance: 'primary',
      leave: 'info',
      code_of_conduct: 'secondary',
      disciplinary: 'error',
      benefits: 'success',
      safety: 'warning',
      other: 'primary'
    };
    return colors[category] || 'primary';
  };

  if (loading && policies.length === 0) {
    return (
      <Box sx={{ display: 'flex', justifyContent: 'center', p: 4 }}>
        <CircularProgress />
      </Box>
    );
  }

  return (
    <Box>
      {error && (
        <Alert severity="error" sx={{ mb: 2 }}>
          {error}
        </Alert>
      )}

      {/* Statistics Cards */}
      {stats && (
        <Box sx={{ display: 'grid', gridTemplateColumns: { xs: '1fr', sm: '1fr 1fr', md: 'repeat(4, 1fr)' }, gap: 3, mb: 3 }}>
          <Box>
            <Card>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: 'primary.main' }}>
                    <Policy />
                  </Avatar>
                  <Box sx={{ ml: 2 }}>
                    <Typography variant="h4">{stats.totalPolicies}</Typography>
                    <Typography color="textSecondary" variant="body2">
                      Total Policies
                    </Typography>
                  </Box>
                </Box>
              </CardContent>
            </Card>
          </Box>

          <Box>
            <Card>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: 'success.main' }}>
                    <CheckCircle />
                  </Avatar>
                  <Box sx={{ ml: 2 }}>
                    <Typography variant="h4">{stats.activePolicies}</Typography>
                    <Typography color="textSecondary" variant="body2">
                      Active Policies
                    </Typography>
                  </Box>
                </Box>
              </CardContent>
            </Card>
          </Box>

          <Box>
            <Card>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: 'warning.main' }}>
                    <Schedule />
                  </Avatar>
                  <Box sx={{ ml: 2 }}>
                    <Typography variant="h4">{stats.draftPolicies}</Typography>
                    <Typography color="textSecondary" variant="body2">
                      Draft Policies
                    </Typography>
                  </Box>
                </Box>
              </CardContent>
            </Card>
          </Box>

          <Box>
            <Card>
              <CardContent>
                <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                  <Avatar sx={{ bgcolor: 'info.main' }}>
                    <Assignment />
                  </Avatar>
                  <Box sx={{ ml: 2 }}>
                    <Typography variant="h4">{stats.pendingAcknowledgments}</Typography>
                    <Typography color="textSecondary" variant="body2">
                      Pending Acknowledgments
                    </Typography>
                  </Box>
                </Box>
              </CardContent>
            </Card>
          </Box>
        </Box>
      )}

      {/* Filters */}
      <Box sx={{ display: 'flex', gap: 2, mb: 3, flexWrap: 'wrap' }}>
        <TextField
          placeholder="Search policies..."
          value={filter.search || ''}
          onChange={(e) => setFilter({ ...filter, search: e.target.value })}
          size="small"
          sx={{ minWidth: 250 }}
          slotProps={{
            input: {
              startAdornment: (
                <InputAdornment position="start">
                  <Search />
                </InputAdornment>
              ),
            }
          }}
        />

        <FormControl size="small" sx={{ minWidth: 150 }}>
          <InputLabel>Category</InputLabel>
          <Select
            value={filter.category || ''}
            onChange={(e) => setFilter({ ...filter, category: e.target.value as HRPolicy['category'] || undefined })}
            label="Category"
          >
            <MenuItem value="">All Categories</MenuItem>
            <MenuItem value="attendance">Attendance</MenuItem>
            <MenuItem value="leave">Leave</MenuItem>
            <MenuItem value="code_of_conduct">Code of Conduct</MenuItem>
            <MenuItem value="disciplinary">Disciplinary</MenuItem>
            <MenuItem value="benefits">Benefits</MenuItem>
            <MenuItem value="safety">Safety</MenuItem>
            <MenuItem value="other">Other</MenuItem>
          </Select>
        </FormControl>

        <FormControl size="small" sx={{ minWidth: 120 }}>
          <InputLabel>Status</InputLabel>
          <Select
            value={filter.status || ''}
            onChange={(e) => setFilter({ ...filter, status: e.target.value as HRPolicy['status'] || undefined })}
            label="Status"
          >
            <MenuItem value="">All Status</MenuItem>
            <MenuItem value="active">Active</MenuItem>
            <MenuItem value="draft">Draft</MenuItem>
            <MenuItem value="archived">Archived</MenuItem>
            <MenuItem value="expired">Expired</MenuItem>
          </Select>
        </FormControl>
      </Box>

      {/* Policies Grid */}
      <Box sx={{ display: 'grid', gridTemplateColumns: { xs: '1fr', md: '1fr 1fr', lg: 'repeat(3, 1fr)' }, gap: 3 }}>
        {policies.map((policy) => (
          <Box key={policy.id}>
            <Card sx={{ height: '100%', display: 'flex', flexDirection: 'column' }}>
              <CardContent sx={{ flexGrow: 1 }}>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'flex-start', mb: 2 }}>
                  <Typography variant="h6" component="h3" noWrap sx={{ flexGrow: 1, mr: 1 }}>
                    {policy.title}
                  </Typography>
                  <Box sx={{ display: 'flex', alignItems: 'center' }}>
                    <Chip
                      icon={getStatusIcon(policy.status)}
                      label={policy.status}
                      size="small"
                      color={getStatusColor(policy.status)}
                      sx={{ mr: 1 }}
                    />
                    <IconButton
                      size="small"
                      onClick={(e) => handleMenuClick(e, policy)}
                    >
                      <MoreVert />
                    </IconButton>
                  </Box>
                </Box>

                <Typography variant="body2" color="text.secondary" sx={{ mb: 2 }}>
                  {policy.description}
                </Typography>

                <Box sx={{ display: 'flex', gap: 1, mb: 2, flexWrap: 'wrap' }}>
                  <Chip
                    label={policy.category.replace('_', ' ')}
                    size="small"
                    color={getCategoryColor(policy.category)}
                    variant="outlined"
                  />
                  <Chip
                    label={`v${policy.version}`}
                    size="small"
                    variant="outlined"
                  />
                  {policy.acknowledgmentRequired && (
                    <Chip
                      label="Requires Ack"
                      size="small"
                      color="info"
                      variant="outlined"
                    />
                  )}
                </Box>

                <Typography variant="body2" color="text.secondary">
                  Effective: {new Date(policy.effectiveDate).toLocaleDateString()}
                </Typography>
                
                {policy.expiryDate && (
                  <Typography variant="body2" color="text.secondary">
                    Expires: {new Date(policy.expiryDate).toLocaleDateString()}
                  </Typography>
                )}

                {policy.acknowledgmentRequired && (
                  <Typography variant="body2" color="text.secondary" sx={{ mt: 1 }}>
                    Acknowledged by: {policy.acknowledgmentCount || 0} employees
                  </Typography>
                )}

                {policy.tags && policy.tags.length > 0 && (
                  <Box sx={{ mt: 2 }}>
                    {policy.tags.slice(0, 3).map((tag) => (
                      <Chip
                        key={tag}
                        label={tag}
                        size="small"
                        variant="filled"
                        sx={{ mr: 0.5, mb: 0.5 }}
                      />
                    ))}
                    {policy.tags.length > 3 && (
                      <Typography variant="caption" color="text.secondary">
                        +{policy.tags.length - 3} more
                      </Typography>
                    )}
                  </Box>
                )}
              </CardContent>

              <CardActions>
                <Button
                  size="small"
                  startIcon={<Visibility />}
                  onClick={() => {
                    setSelectedPolicy(policy);
                    setDialogType('view');
                  }}
                >
                  View
                </Button>
                <Button
                  size="small"
                  startIcon={<Edit />}
                  onClick={() => {
                    setSelectedPolicy(policy);
                    setDialogType('edit');
                  }}
                >
                  Edit
                </Button>
              </CardActions>
            </Card>
          </Box>
        ))}
      </Box>

      {policies.length === 0 && !loading && (
        <Box sx={{ textAlign: 'center', py: 8 }}>
          <Policy sx={{ fontSize: 64, color: 'text.secondary', mb: 2 }} />
          <Typography variant="h6" color="text.secondary" gutterBottom>
            No policies found
          </Typography>
          <Typography variant="body2" color="text.secondary" sx={{ mb: 3 }}>
            Create your first HR policy to get started
          </Typography>
          <Button variant="contained" startIcon={<Add />} onClick={handleCreatePolicy}>
            Create Policy
          </Button>
        </Box>
      )}

      {/* Floating Action Button */}
      <Fab
        color="primary"
        aria-label="add"
        sx={{ position: 'fixed', bottom: 16, right: 16 }}
        onClick={handleCreatePolicy}
      >
        <Add />
      </Fab>

      {/* Context Menu */}
      <Menu
        anchorEl={anchorEl}
        open={Boolean(anchorEl)}
        onClose={handleMenuClose}
      >
        <MenuItem onClick={handleViewPolicy}>
          <ListItemIcon>
            <Visibility />
          </ListItemIcon>
          <ListItemText>View Policy</ListItemText>
        </MenuItem>
        <MenuItem onClick={handleEditPolicy}>
          <ListItemIcon>
            <Edit />
          </ListItemIcon>
          <ListItemText>Edit Policy</ListItemText>
        </MenuItem>
        <MenuItem onClick={() => {}}>
          <ListItemIcon>
            <Download />
          </ListItemIcon>
          <ListItemText>Download PDF</ListItemText>
        </MenuItem>
        <MenuItem onClick={handleDeletePolicy} sx={{ color: 'error.main' }}>
          <ListItemIcon>
            <Delete sx={{ color: 'error.main' }} />
          </ListItemIcon>
          <ListItemText>Delete Policy</ListItemText>
        </MenuItem>
      </Menu>

      {/* Policy Dialog */}
      <PolicyDialog
        open={dialogType === 'create' || dialogType === 'edit'}
        onClose={handleDialogClose}
        onSave={handleDialogSave}
        policy={selectedPolicy}
      />

      {/* Delete Confirmation Dialog */}
      <Dialog open={deleteConfirmOpen} onClose={() => setDeleteConfirmOpen(false)}>
        <DialogTitle>Delete Policy</DialogTitle>
        <DialogContent>
          <DialogContentText>
            Are you sure you want to delete the policy "{policyToDelete?.title}"? This action cannot be undone.
          </DialogContentText>
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setDeleteConfirmOpen(false)}>Cancel</Button>
          <Button onClick={confirmDelete} color="error" autoFocus>
            Delete
          </Button>
        </DialogActions>
      </Dialog>

      {/* Loader Overlay */}
      <LoaderOverlay
        open={Boolean(operationLoading)}
        message={operationLoading || 'Loading...'}
      />
    </Box>
  );
};