import React, { useState, useEffect } from 'react';
import {
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  Button,
  TextField,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Box,
  Typography,
  IconButton,
  Alert,
  CircularProgress,
  Switch,
  FormControlLabel,
  Chip,
  Stack,
} from '@mui/material';
import {
  Close,
  Save,
} from '@mui/icons-material';
import { HRPolicy, PolicyCreateRequest, PolicyUpdateRequest } from '../../types';
import { complianceService } from '../../services';

interface PolicyDialogProps {
  open: boolean;
  onClose: () => void;
  onSave: (policy: HRPolicy) => void;
  policy?: HRPolicy | null;
}

export const PolicyDialog: React.FC<PolicyDialogProps> = ({
  open,
  onClose,
  onSave,
  policy,
}) => {
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState('');
  const [formData, setFormData] = useState<{
    title: string;
    description: string;
    category: HRPolicy['category'];
    content: string;
    effectiveDate: string;
    expiryDate: string;
    acknowledgmentRequired: boolean;
    applicableTo: HRPolicy['applicableTo'];
    applicableIds: string[];
    tags: string[];
    newTag: string;
  }>({
    title: '',
    description: '',
    category: 'other',
    content: '',
    effectiveDate: new Date().toISOString().split('T')[0],
    expiryDate: '',
    acknowledgmentRequired: true,
    applicableTo: 'all',
    applicableIds: [],
    tags: [],
    newTag: '',
  });

  useEffect(() => {
    if (policy) {
      setFormData({
        title: policy.title,
        description: policy.description,
        category: policy.category,
        content: policy.content,
        effectiveDate: policy.effectiveDate,
        expiryDate: policy.expiryDate || '',
        acknowledgmentRequired: policy.acknowledgmentRequired,
        applicableTo: policy.applicableTo,
        applicableIds: policy.applicableIds || [],
        tags: policy.tags || [],
        newTag: '',
      });
    } else {
      setFormData({
        title: '',
        description: '',
        category: 'other',
        content: '',
        effectiveDate: new Date().toISOString().split('T')[0],
        expiryDate: '',
        acknowledgmentRequired: true,
        applicableTo: 'all',
        applicableIds: [],
        tags: [],
        newTag: '',
      });
    }
    setError('');
  }, [policy, open]);

  const handleInputChange = (field: keyof typeof formData, value: any) => {
    setFormData({ ...formData, [field]: value });
  };

  const handleAddTag = () => {
    if (formData.newTag.trim() && !formData.tags.includes(formData.newTag.trim())) {
      setFormData({
        ...formData,
        tags: [...formData.tags, formData.newTag.trim()],
        newTag: '',
      });
    }
  };

  const handleRemoveTag = (tagToRemove: string) => {
    setFormData({
      ...formData,
      tags: formData.tags.filter(tag => tag !== tagToRemove),
    });
  };

  const handleSave = async () => {
    setLoading(true);
    setError('');

    try {
      // Validate form
      const validation = await complianceService.validatePolicy({
        title: formData.title,
        description: formData.description,
        category: formData.category,
        content: formData.content,
        effectiveDate: formData.effectiveDate,
        expiryDate: formData.expiryDate || undefined,
        acknowledgmentRequired: formData.acknowledgmentRequired,
        applicableTo: formData.applicableTo,
        applicableIds: formData.applicableIds,
        tags: formData.tags,
      });

      if (!validation.isValid) {
        setError(validation.errors.join(', '));
        setLoading(false);
        return;
      }

      let result: HRPolicy | null;

      if (policy) {
        // Update policy
        const updateData: PolicyUpdateRequest = {
          title: formData.title,
          description: formData.description,
          category: formData.category,
          content: formData.content,
          effectiveDate: formData.effectiveDate,
          expiryDate: formData.expiryDate || undefined,
          acknowledgmentRequired: formData.acknowledgmentRequired,
          applicableTo: formData.applicableTo,
          applicableIds: formData.applicableIds,
          tags: formData.tags,
        };

        result = await complianceService.updatePolicy(policy.id, updateData);
      } else {
        // Create policy
        const createData: PolicyCreateRequest = {
          title: formData.title,
          description: formData.description,
          category: formData.category,
          content: formData.content,
          effectiveDate: formData.effectiveDate,
          expiryDate: formData.expiryDate || undefined,
          acknowledgmentRequired: formData.acknowledgmentRequired,
          applicableTo: formData.applicableTo,
          applicableIds: formData.applicableIds,
          tags: formData.tags,
        };

        result = await complianceService.createPolicy(createData);
      }

      if (result) {
        onSave(result);
        onClose();
      } else {
        setError('Failed to save policy');
      }
    } catch (err: any) {
      setError(err.message || 'Failed to save policy');
    } finally {
      setLoading(false);
    }
  };

  const isFormValid = () => {
    return formData.title.trim().length >= 3 && 
           formData.description.trim().length >= 10 &&
           formData.content.trim().length >= 50;
  };

  return (
    <Dialog
      open={open}
      onClose={onClose}
      maxWidth="md"
      fullWidth
      slotProps={{
        paper: { sx: { minHeight: '80vh', maxHeight: '90vh' } }
      }}
    >
      <DialogTitle sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
        <Typography variant="h6">
          {policy ? 'Edit Policy' : 'Create New Policy'}
        </Typography>
        <IconButton onClick={onClose}>
          <Close />
        </IconButton>
      </DialogTitle>

      <DialogContent sx={{ overflow: 'auto' }}>
        {error && (
          <Alert severity="error" sx={{ mb: 2 }}>
            {error}
          </Alert>
        )}

        <Box sx={{ display: 'flex', flexDirection: 'column', gap: 3, pt: 2 }}>
          {/* Basic Information */}
          <Box>
            <Typography variant="subtitle1" gutterBottom fontWeight="medium">
              Basic Information
            </Typography>
            
            <Box sx={{ display: 'flex', flexDirection: 'column', gap: 2 }}>
              <TextField
                label="Policy Title"
                value={formData.title}
                onChange={(e) => handleInputChange('title', e.target.value)}
                fullWidth
                required
              />

              <TextField
                label="Description"
                value={formData.description}
                onChange={(e) => handleInputChange('description', e.target.value)}
                multiline
                rows={2}
                fullWidth
                required
              />

              <FormControl fullWidth required>
                <InputLabel>Category</InputLabel>
                <Select
                  value={formData.category}
                  onChange={(e) => handleInputChange('category', e.target.value)}
                  label="Category"
                >
                  <MenuItem value="attendance">Attendance</MenuItem>
                  <MenuItem value="leave">Leave</MenuItem>
                  <MenuItem value="code_of_conduct">Code of Conduct</MenuItem>
                  <MenuItem value="disciplinary">Disciplinary</MenuItem>
                  <MenuItem value="benefits">Benefits</MenuItem>
                  <MenuItem value="safety">Safety</MenuItem>
                  <MenuItem value="other">Other</MenuItem>
                </Select>
              </FormControl>

              <Box sx={{ display: 'flex', gap: 2 }}>
                <TextField
                  label="Effective Date"
                  type="date"
                  value={formData.effectiveDate}
                  onChange={(e) => handleInputChange('effectiveDate', e.target.value)}
                  fullWidth
                  required
                />

                <TextField
                  label="Expiry Date (Optional)"
                  type="date"
                  value={formData.expiryDate}
                  onChange={(e) => handleInputChange('expiryDate', e.target.value)}
                  fullWidth
                />
              </Box>
            </Box>
          </Box>

          {/* Policy Content */}
          <Box>
            <Typography variant="subtitle1" gutterBottom fontWeight="medium">
              Policy Content
            </Typography>
            
            <TextField
              label="Policy Content"
              value={formData.content}
              onChange={(e) => handleInputChange('content', e.target.value)}
              multiline
              rows={8}
              fullWidth
              required
              placeholder="Enter the detailed policy content here..."
              helperText="HTML tags are supported for rich text formatting"
            />
          </Box>

          {/* Applicability */}
          <Box>
            <Typography variant="subtitle1" gutterBottom fontWeight="medium">
              Applicability
            </Typography>
            
            <FormControl fullWidth>
              <InputLabel>Applicable To</InputLabel>
              <Select
                value={formData.applicableTo}
                onChange={(e) => handleInputChange('applicableTo', e.target.value)}
                label="Applicable To"
              >
                <MenuItem value="all">All Employees</MenuItem>
                <MenuItem value="department">Specific Departments</MenuItem>
                <MenuItem value="role">Specific Roles</MenuItem>
                <MenuItem value="specific">Specific Employees</MenuItem>
              </Select>
            </FormControl>

            <FormControlLabel
              control={
                <Switch
                  checked={formData.acknowledgmentRequired}
                  onChange={(e) => handleInputChange('acknowledgmentRequired', e.target.checked)}
                />
              }
              label="Require Employee Acknowledgment"
              sx={{ mt: 1 }}
            />
          </Box>

          {/* Tags */}
          <Box>
            <Typography variant="subtitle1" gutterBottom fontWeight="medium">
              Tags
            </Typography>
            
            <Box sx={{ display: 'flex', gap: 1, mb: 2 }}>
              <TextField
                label="Add Tag"
                value={formData.newTag}
                onChange={(e) => handleInputChange('newTag', e.target.value)}
                size="small"
                onKeyDown={(e) => e.key === 'Enter' && handleAddTag()}
              />
              <Button onClick={handleAddTag} variant="outlined" size="small">
                Add
              </Button>
            </Box>

            <Stack direction="row" spacing={1} flexWrap="wrap">
              {formData.tags.map((tag) => (
                <Chip
                  key={tag}
                  label={tag}
                  onDelete={() => handleRemoveTag(tag)}
                  size="small"
                  color="primary"
                  variant="outlined"
                />
              ))}
            </Stack>
          </Box>
        </Box>
      </DialogContent>

      <DialogActions sx={{ p: 3 }}>
        <Button onClick={onClose}>
          Cancel
        </Button>
        <Button
          variant="contained"
          startIcon={loading ? <CircularProgress size={16} /> : <Save />}
          onClick={handleSave}
          disabled={!isFormValid() || loading}
        >
          {loading ? 'Saving...' : policy ? 'Update Policy' : 'Create Policy'}
        </Button>
      </DialogActions>
    </Dialog>
  );
};