import { LeaveRequest } from '../index';

// Leave service specific types
export interface LeaveFilter {
  employeeId?: string;
  department?: string;
  type?: LeaveRequest['type'];
  status?: LeaveRequest['status'];
  dateRange?: {
    startDate: string;
    endDate: string;
  };
  appliedDateRange?: {
    startDate: string;
    endDate: string;
  };
  approvedBy?: string;
  daysRange?: {
    min: number;
    max: number;
  };
}

export interface LeaveSortOptions {
  field: 'appliedDate' | 'startDate' | 'endDate' | 'days' | 'status' | 'type';
  order: 'asc' | 'desc';
}

export interface LeavePaginationOptions {
  page: number;
  limit: number;
  sort?: LeaveSortOptions;
}

export interface LeaveListResponse {
  leaves: LeaveRequest[];
  total: number;
  page: number;
  limit: number;
  totalPages: number;
}

export interface LeaveCreateRequest extends Omit<LeaveRequest, 'id' | 'appliedDate' | 'status'> {}

export interface LeaveUpdateRequest extends Partial<Omit<LeaveRequest, 'id' | 'appliedDate'>> {}

export interface LeaveApprovalRequest {
  leaveId: string;
  approvedBy: string;
  comments?: string;
  action: 'approve' | 'reject';
}

export interface LeaveBulkApprovalRequest {
  leaveIds: string[];
  approvedBy: string;
  action: 'approve' | 'reject';
  comments?: string;
}

export interface LeaveStats {
  total: number;
  pending: number;
  approved: number;
  rejected: number;
  totalApprovedDays: number;
  byType: {
    [K in LeaveRequest['type']]: number;
  };
  byDepartment: Array<{
    department: string;
    total: number;
    approved: number;
    avgDays: number;
  }>;
}

export interface LeaveBalance {
  employeeId: string;
  employeeName: string;
  leaveTypes: Array<{
    type: LeaveRequest['type'];
    allocated: number;
    used: number;
    remaining: number;
    carriedForward: number;
  }>;
  totalAllocated: number;
  totalUsed: number;
  totalRemaining: number;
}

export interface LeavePolicy {
  id: string;
  name: string;
  description: string;
  leaveType: LeaveRequest['type'];
  allocationType: 'annual' | 'monthly' | 'accrual';
  allocation: number; // days per period
  maxCarryForward: number;
  minNotice: number; // days
  maxConsecutiveDays: number;
  requiresApproval: boolean;
  applicableRoles: string[];
  blackoutDates: string[];
  isActive: boolean;
}

export interface LeaveCalendar {
  month: number;
  year: number;
  leaves: Array<{
    date: string;
    leaves: Array<{
      employeeId: string;
      employeeName: string;
      type: LeaveRequest['type'];
      status: LeaveRequest['status'];
    }>;
  }>;
  holidays: Array<{
    date: string;
    name: string;
    type: 'public' | 'company';
  }>;
  departmentCoverage: Array<{
    department: string;
    date: string;
    totalEmployees: number;
    onLeave: number;
    availabilityPercentage: number;
  }>;
}

export interface LeaveConflict {
  date: string;
  department: string;
  conflictingLeaves: Array<{
    employeeId: string;
    employeeName: string;
    leaveId: string;
    type: LeaveRequest['type'];
  }>;
  impactLevel: 'low' | 'medium' | 'high' | 'critical';
  suggestions: string[];
}

export interface LeaveReport {
  reportType: 'summary' | 'detailed' | 'trend' | 'utilization';
  period: {
    startDate: string;
    endDate: string;
  };
  filters?: LeaveFilter;
  data: any; // Flexible data structure based on report type
  generatedAt: string;
  generatedBy: string;
}

export interface LeaveNotification {
  id: string;
  type: 'application' | 'approval' | 'rejection' | 'reminder' | 'balance_low';
  employeeId: string;
  leaveId?: string;
  title: string;
  message: string;
  isRead: boolean;
  createdAt: string;
  actionRequired: boolean;
  actionUrl?: string;
}

export interface LeaveEscalation {
  leaveId: string;
  currentApprover: string;
  escalationLevel: number;
  escalatedTo: string;
  reason: string;
  escalatedAt: string;
  isResolved: boolean;
}

export interface LeaveRecurring {
  id: string;
  employeeId: string;
  type: LeaveRequest['type'];
  pattern: 'weekly' | 'monthly' | 'yearly';
  startDate: string;
  endDate?: string; // null for indefinite
  daysPerOccurrence: number;
  reason: string;
  isActive: boolean;
  nextOccurrence: string;
}

export interface LeaveAdjustment {
  id: string;
  employeeId: string;
  leaveType: LeaveRequest['type'];
  adjustmentType: 'add' | 'deduct';
  days: number;
  reason: string;
  adjustedBy: string;
  adjustedAt: string;
  relatedLeaveId?: string;
}