import { 
  UserProfile,
  UserCreateRequest,
  UserUpdateRequest,
  UserFilter,
  UserListResponse,
  UserStats,
  UserBulkOperation,
  UserBulkOperationResult,
  UserPaginationOptions,
  UserPermission,
  Role,
  RoleCreateRequest,
  RoleUpdateRequest,
  RoleFilter,
  RoleListResponse,
  RoleStats,
  RolePaginationOptions,
  Permission,
  PermissionGroup
} from '../types';

// Mock data for development
const mockUsers: UserProfile[] = [
  {
    id: '1',
    email: 'admin@hr.com',
    name: 'Admin User',
    role: 'admin',
    employeeId: 'EMP001',
    department: 'IT',
    position: 'System Administrator',
    phone: '+1-555-0101',
    isActive: true,
    lastLogin: '2023-12-01T10:00:00Z',
    createdAt: '2023-01-01T00:00:00Z',
    updatedAt: '2023-12-01T10:00:00Z',
    preferences: {
      theme: 'dark',
      language: 'en',
      timezone: 'America/New_York',
      dateFormat: 'MM/DD/YYYY',
      notifications: {
        email: true,
        browser: true,
        mobile: false,
        frequency: 'immediate'
      },
      dashboard: {
        widgets: [
          { id: 'employee-stats', position: 1, isVisible: true },
          { id: 'attendance-overview', position: 2, isVisible: true }
        ],
        layout: 'grid'
      }
    },
    permissions: [
      {
        id: '1',
        module: 'admin',
        actions: ['create', 'read', 'update', 'delete'],
      }
    ]
  },
  {
    id: '2',
    email: 'hr@company.com',
    name: 'HR Manager',
    role: 'hr',
    employeeId: 'EMP002',
    department: 'Human Resources',
    position: 'HR Manager',
    phone: '+1-555-0102',
    isActive: true,
    lastLogin: '2023-12-01T09:30:00Z',
    createdAt: '2023-01-15T00:00:00Z',
    updatedAt: '2023-12-01T09:30:00Z',
    preferences: {
      theme: 'light',
      language: 'en',
      timezone: 'America/New_York',
      dateFormat: 'MM/DD/YYYY',
      notifications: {
        email: true,
        browser: true,
        mobile: true,
        frequency: 'daily'
      },
      dashboard: {
        widgets: [
          { id: 'leave-requests', position: 1, isVisible: true },
          { id: 'employee-stats', position: 2, isVisible: true }
        ],
        layout: 'grid'
      }
    },
    permissions: [
      {
        id: '2',
        module: 'employees',
        actions: ['create', 'read', 'update'],
      },
      {
        id: '3',
        module: 'leaves',
        actions: ['read', 'approve'],
      }
    ]
  }
];

const mockRoles: Role[] = [
  {
    id: 'role-1',
    name: 'Super Administrator',
    description: 'Full system access with all permissions',
    permissions: [
      {
        id: 'perm-1',
        name: 'System Administration',
        description: 'Full system control',
        resource: 'system',
        action: 'execute',
        category: 'system'
      },
      {
        id: 'perm-2',
        name: 'User Management',
        description: 'Manage all users',
        resource: 'users',
        action: 'create',
        category: 'system'
      }
    ],
    isSystem: true,
    isActive: true,
    createdAt: new Date('2023-01-01'),
    updatedAt: new Date('2023-01-01'),
    createdBy: 'system',
    userCount: 1
  },
  {
    id: 'role-2',
    name: 'HR Manager',
    description: 'Human Resources management permissions',
    permissions: [
      {
        id: 'perm-3',
        name: 'Employee Management',
        description: 'Manage employee records',
        resource: 'employees',
        action: 'create',
        category: 'hr'
      },
      {
        id: 'perm-4',
        name: 'Leave Management',
        description: 'Process leave requests',
        resource: 'leaves',
        action: 'update',
        category: 'hr'
      }
    ],
    isSystem: false,
    isActive: true,
    createdAt: new Date('2023-01-01'),
    updatedAt: new Date('2023-01-01'),
    createdBy: 'admin',
    userCount: 5
  },
  {
    id: 'role-3',
    name: 'Employee',
    description: 'Standard employee permissions',
    permissions: [
      {
        id: 'perm-5',
        name: 'View Own Data',
        description: 'View personal information',
        resource: 'employees',
        action: 'read',
        category: 'employee'
      },
      {
        id: 'perm-6',
        name: 'Submit Leave',
        description: 'Submit leave requests',
        resource: 'leaves',
        action: 'create',
        category: 'employee'
      }
    ],
    isSystem: true,
    isActive: true,
    createdAt: new Date('2023-01-01'),
    updatedAt: new Date('2023-01-01'),
    createdBy: 'system',
    userCount: 50
  }
];

// Extract all permissions from roles and groups for easier lookup
const mockPermissions: UserPermission[] = [
  { id: 'perm-1', module: 'admin', actions: ['read', 'create', 'update', 'delete'] },
  { id: 'perm-2', module: 'employees', actions: ['read', 'create', 'update', 'delete'] },
  { id: 'perm-3', module: 'employees', actions: ['read', 'create', 'update'] },
  { id: 'perm-4', module: 'leaves', actions: ['read', 'update', 'approve'] },
  { id: 'perm-5', module: 'employees', actions: ['read'] },
  { id: 'perm-6', module: 'leaves', actions: ['create'] },
];

const mockPermissionGroups: PermissionGroup[] = [
  {
    id: 'group-1',
    name: 'System Administration',
    description: 'System-level administrative permissions',
    category: 'system',
    permissions: [
      {
        id: 'perm-1',
        name: 'System Configuration',
        description: 'Configure system settings',
        resource: 'system',
        action: 'update',
        category: 'system'
      },
      {
        id: 'perm-2',
        name: 'User Management',
        description: 'Manage system users',
        resource: 'users',
        action: 'create',
        category: 'system'
      }
    ]
  },
  {
    id: 'group-2',
    name: 'Employee Management',
    description: 'Employee-related permissions',
    category: 'hr',
    permissions: [
      {
        id: 'perm-3',
        name: 'Create Employees',
        description: 'Add new employees',
        resource: 'employees',
        action: 'create',
        category: 'hr'
      },
      {
        id: 'perm-4',
        name: 'Update Employees',
        description: 'Modify employee records',
        resource: 'employees',
        action: 'update',
        category: 'hr'
      }
    ]
  }
];

class UserManagementService {
  // User CRUD Operations
  async getAllUsers(filter?: UserFilter, pagination?: UserPaginationOptions): Promise<UserListResponse> {
    await new Promise(resolve => setTimeout(resolve, 500)); // Simulate API call
    
    let filteredUsers = [...mockUsers];
    
    if (filter?.search) {
      const searchTerm = filter.search.toLowerCase();
      filteredUsers = filteredUsers.filter(user => 
        user.name.toLowerCase().includes(searchTerm) ||
        user.email.toLowerCase().includes(searchTerm) ||
        user.department?.toLowerCase().includes(searchTerm)
      );
    }
    
    if (filter?.role) {
      filteredUsers = filteredUsers.filter(user => user.role === filter.role);
    }
    
    if (filter?.isActive !== undefined) {
      filteredUsers = filteredUsers.filter(user => user.isActive === filter.isActive);
    }
    
    const page = pagination?.page || 1;
    const limit = pagination?.limit || 10;
    const startIndex = (page - 1) * limit;
    const endIndex = startIndex + limit;
    
    const paginatedUsers = filteredUsers.slice(startIndex, endIndex);
    
    return {
      users: paginatedUsers,
      total: filteredUsers.length,
      page,
      limit,
      totalPages: Math.ceil(filteredUsers.length / limit)
    };
  }

  async getUserById(id: string): Promise<UserProfile | null> {
    await new Promise(resolve => setTimeout(resolve, 300));
    return mockUsers.find(user => user.id === id) || null;
  }

  async getUserByEmail(email: string): Promise<UserProfile | null> {
    await new Promise(resolve => setTimeout(resolve, 300));
    return mockUsers.find(user => user.email === email) || null;
  }

  async createUser(userData: UserCreateRequest): Promise<UserProfile> {
    await new Promise(resolve => setTimeout(resolve, 800));
    
    const newUser: UserProfile = {
      id: `user-${Date.now()}`,
      email: userData.email,
      name: userData.name,
      role: userData.role,
      employeeId: userData.employeeId,
      department: userData.department,
      position: userData.position,
      phone: userData.phone,
      isActive: true,
      lastLogin: undefined,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString(),
      preferences: {
        theme: 'light',
        language: 'en',
        timezone: 'America/New_York',
        dateFormat: 'MM/DD/YYYY',
        notifications: {
          email: true,
          browser: true,
          mobile: false,
          frequency: 'daily'
        },
        dashboard: {
          widgets: [],
          layout: 'grid'
        }
      },
      permissions: userData.permissions?.map(permId => ({
        id: permId,
        module: 'employees',
        actions: ['read']
      })) || []
    };
    
    mockUsers.push(newUser);
    return newUser;
  }

  async updateUser(id: string, updates: UserUpdateRequest): Promise<UserProfile | null> {
    await new Promise(resolve => setTimeout(resolve, 600));
    
    const userIndex = mockUsers.findIndex(user => user.id === id);
    if (userIndex === -1) return null;
    
    const existingUser = mockUsers[userIndex];
    
    // Convert permission IDs to permission objects if needed
    let updatedPermissions = existingUser.permissions;
    if (updates.permissions) {
      updatedPermissions = updates.permissions.map((permId: string) => {
        const permission = mockPermissions.find(p => p.id === permId);
        return permission || existingUser.permissions.find(p => p.id === permId) || {
          id: permId,
          module: 'employees' as const,
          actions: ['read' as const]
        };
      });
    }
    
    const updatedUser: UserProfile = {
      ...existingUser,
      ...updates,
      permissions: updatedPermissions,
      preferences: updates.preferences 
        ? { ...existingUser.preferences, ...updates.preferences }
        : existingUser.preferences,
      updatedAt: new Date().toISOString()
    };
    
    mockUsers[userIndex] = updatedUser;
    return updatedUser;
  }

  async deleteUser(id: string): Promise<boolean> {
    await new Promise(resolve => setTimeout(resolve, 400));
    
    const userIndex = mockUsers.findIndex(user => user.id === id);
    if (userIndex === -1) return false;
    
    mockUsers.splice(userIndex, 1);
    return true;
  }

  async getUserStats(filter?: UserFilter): Promise<UserStats> {
    await new Promise(resolve => setTimeout(resolve, 400));
    
    let users = [...mockUsers];
    
    if (filter?.department) {
      users = users.filter(user => user.department === filter.department);
    }
    
    return {
      total: users.length,
      active: users.filter(user => user.isActive).length,
      inactive: users.filter(user => !user.isActive).length,
      byRole: {
        admin: users.filter(user => user.role === 'admin').length,
        hr: users.filter(user => user.role === 'hr').length,
        employee: users.filter(user => user.role === 'employee').length,
      },
      recentLogins: users.filter(user => {
        if (!user.lastLogin) return false;
        const weekAgo = new Date();
        weekAgo.setDate(weekAgo.getDate() - 7);
        return new Date(user.lastLogin) > weekAgo;
      }).length,
      sessionsActive: Math.floor(users.length * 0.3), // Mock
      avgSessionDuration: 45 // Mock
    };
  }

  async bulkUpdateUsers(operation: UserBulkOperation): Promise<UserBulkOperationResult> {
    await new Promise(resolve => setTimeout(resolve, 1000));
    
    let processedCount = 0;
    let failedCount = 0;
    const errors: Array<{ userId: string; error: string }> = [];
    
    for (const userId of operation.userIds) {
      const userIndex = mockUsers.findIndex(user => user.id === userId);
      
      if (userIndex === -1) {
        failedCount++;
        errors.push({ userId, error: 'User not found' });
        continue;
      }
      
      try {
        switch (operation.action) {
          case 'activate':
            mockUsers[userIndex].isActive = true;
            break;
          case 'deactivate':
            mockUsers[userIndex].isActive = false;
            break;
          case 'update_role':
            if (operation.params?.newRole) {
              mockUsers[userIndex].role = operation.params.newRole;
            }
            break;
          case 'delete':
            mockUsers.splice(userIndex, 1);
            break;
        }
        processedCount++;
      } catch (error) {
        failedCount++;
        errors.push({ userId, error: 'Update failed' });
      }
    }
    
    return {
      success: failedCount === 0,
      processedCount,
      failedCount,
      errors
    };
  }

  // Role Management Methods
  async getAllRoles(filter?: RoleFilter, pagination?: RolePaginationOptions): Promise<RoleListResponse> {
    await new Promise(resolve => setTimeout(resolve, 400));
    
    let filteredRoles = [...mockRoles];
    
    if (filter?.search) {
      const searchTerm = filter.search.toLowerCase();
      filteredRoles = filteredRoles.filter(role => 
        role.name.toLowerCase().includes(searchTerm) ||
        role.description.toLowerCase().includes(searchTerm)
      );
    }
    
    if (filter?.isActive !== undefined) {
      filteredRoles = filteredRoles.filter(role => role.isActive === filter.isActive);
    }
    
    if (filter?.isSystem !== undefined) {
      filteredRoles = filteredRoles.filter(role => role.isSystem === filter.isSystem);
    }
    
    const page = pagination?.page || 1;
    const limit = pagination?.limit || 10;
    const startIndex = (page - 1) * limit;
    const endIndex = startIndex + limit;
    
    return {
      roles: filteredRoles.slice(startIndex, endIndex),
      total: filteredRoles.length,
      page,
      limit,
      totalPages: Math.ceil(filteredRoles.length / limit)
    };
  }

  async getRoleById(id: string): Promise<Role | null> {
    await new Promise(resolve => setTimeout(resolve, 200));
    return mockRoles.find(role => role.id === id) || null;
  }

  async createRole(roleData: RoleCreateRequest): Promise<Role> {
    await new Promise(resolve => setTimeout(resolve, 600));
    
    const permissions = mockPermissionGroups
      .flatMap(group => group.permissions)
      .filter(perm => roleData.permissionIds.includes(perm.id));
    
    const newRole: Role = {
      id: `role-${Date.now()}`,
      name: roleData.name,
      description: roleData.description,
      permissions,
      isSystem: false,
      isActive: true,
      createdAt: new Date(),
      updatedAt: new Date(),
      createdBy: 'current-user',
      userCount: 0
    };
    
    mockRoles.push(newRole);
    return newRole;
  }

  async updateRole(id: string, updates: RoleUpdateRequest): Promise<Role | null> {
    await new Promise(resolve => setTimeout(resolve, 500));
    
    const roleIndex = mockRoles.findIndex(role => role.id === id);
    if (roleIndex === -1) return null;
    
    let permissions = mockRoles[roleIndex].permissions;
    
    if (updates.permissionIds) {
      permissions = mockPermissionGroups
        .flatMap(group => group.permissions)
        .filter(perm => updates.permissionIds!.includes(perm.id));
    }
    
    const updatedRole: Role = {
      ...mockRoles[roleIndex],
      ...updates,
      permissions,
      updatedAt: new Date()
    };
    
    mockRoles[roleIndex] = updatedRole;
    return updatedRole;
  }

  async deleteRole(id: string): Promise<boolean> {
    await new Promise(resolve => setTimeout(resolve, 300));
    
    const roleIndex = mockRoles.findIndex(role => role.id === id);
    if (roleIndex === -1) return false;
    
    if (mockRoles[roleIndex].isSystem) {
      throw new Error('Cannot delete system role');
    }
    
    mockRoles.splice(roleIndex, 1);
    return true;
  }

  async getRoleStats(): Promise<RoleStats> {
    await new Promise(resolve => setTimeout(resolve, 300));
    
    return {
      total: mockRoles.length,
      system: mockRoles.filter(role => role.isSystem).length,
      custom: mockRoles.filter(role => !role.isSystem).length,
      active: mockRoles.filter(role => role.isActive).length,
      inactive: mockRoles.filter(role => !role.isActive).length,
    };
  }

  async getPermissionGroups(): Promise<PermissionGroup[]> {
    await new Promise(resolve => setTimeout(resolve, 200));
    return [...mockPermissionGroups];
  }

  async getAllPermissions(): Promise<Permission[]> {
    await new Promise(resolve => setTimeout(resolve, 200));
    return mockPermissionGroups.flatMap(group => group.permissions);
  }

  // User Role Assignment
  async assignRole(userId: string, roleId: string): Promise<boolean> {
    await new Promise(resolve => setTimeout(resolve, 400));
    
    const userIndex = mockUsers.findIndex(user => user.id === userId);
    const role = mockRoles.find(role => role.id === roleId);
    
    if (userIndex === -1 || !role) return false;
    
    // For simplicity, we'll just update the user's role property
    // In a real system, you'd handle multiple roles properly
    mockUsers[userIndex].role = role.name.toLowerCase() as any;
    mockUsers[userIndex].updatedAt = new Date().toISOString();
    
    return true;
  }

  // Validation Methods
  async validateUser(user: UserCreateRequest): Promise<{ isValid: boolean; errors: string[] }> {
    const errors: string[] = [];
    
    if (!user.email || !user.email.includes('@')) {
      errors.push('Valid email is required');
    }
    
    if (mockUsers.some(existingUser => existingUser.email === user.email)) {
      errors.push('Email already exists');
    }
    
    if (!user.name || user.name.trim().length < 2) {
      errors.push('Name must be at least 2 characters');
    }
    
    if (!user.password || user.password.length < 6) {
      errors.push('Password must be at least 6 characters');
    }
    
    return {
      isValid: errors.length === 0,
      errors
    };
  }

  async validateRole(role: RoleCreateRequest): Promise<{ isValid: boolean; errors: string[] }> {
    const errors: string[] = [];
    
    if (!role.name || role.name.trim().length < 2) {
      errors.push('Role name must be at least 2 characters');
    }
    
    if (mockRoles.some(existingRole => existingRole.name.toLowerCase() === role.name.toLowerCase())) {
      errors.push('Role name already exists');
    }
    
    if (!role.permissionIds || role.permissionIds.length === 0) {
      errors.push('At least one permission is required');
    }
    
    return {
      isValid: errors.length === 0,
      errors
    };
  }
}

export const userManagementService = new UserManagementService();