export interface Notification {
  id: string;
  userId: string;
  type: 'info' | 'warning' | 'error' | 'success';
  category: 'leave' | 'payroll' | 'attendance' | 'system' | 'reminder';
  title: string;
  message: string;
  read: boolean;
  createdAt: string;
  actionUrl?: string;
  metadata?: any;
}

// Mock notifications data
const mockNotifications: Notification[] = [
  {
    id: '1',
    userId: '1',
    type: 'warning',
    category: 'leave',
    title: 'Pending Leave Approval',
    message: '12 leave requests are pending approval',
    read: false,
    createdAt: new Date().toISOString(),
    actionUrl: '/leaves?filter=pending',
  },
  {
    id: '2',
    userId: '1',
    type: 'info',
    category: 'payroll',
    title: 'Payroll Processing',
    message: 'Monthly payroll for June is ready for processing',
    read: false,
    createdAt: new Date(Date.now() - 2 * 60 * 60 * 1000).toISOString(),
    actionUrl: '/payroll?status=processed',
  },
  {
    id: '3',
    userId: '1',
    type: 'success',
    category: 'attendance',
    title: 'Attendance Rate',
    message: 'Great job! Attendance rate this week is 94.5%',
    read: true,
    createdAt: new Date(Date.now() - 24 * 60 * 60 * 1000).toISOString(),
  },
  {
    id: '4',
    userId: '2',
    type: 'error',
    category: 'system',
    title: 'System Alert',
    message: 'Failed to sync attendance data for 3 employees',
    read: false,
    createdAt: new Date(Date.now() - 6 * 60 * 60 * 1000).toISOString(),
    actionUrl: '/attendance',
  },
];

let notifications = [...mockNotifications];

export const notificationService = {
  // Get all notifications for a user
  getUserNotifications: async (userId: string, unreadOnly?: boolean): Promise<Notification[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        let userNotifications = notifications.filter(n => n.userId === userId);
        
        if (unreadOnly) {
          userNotifications = userNotifications.filter(n => !n.read);
        }
        
        // Sort by created date (newest first)
        userNotifications.sort((a, b) => 
          new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime()
        );
        
        resolve(userNotifications);
      }, 300);
    });
  },

  // Get notifications by category
  getNotificationsByCategory: async (userId: string, category: string): Promise<Notification[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const filtered = notifications.filter(n => 
          n.userId === userId && n.category === category
        );
        resolve(filtered);
      }, 300);
    });
  },

  // Create new notification
  createNotification: async (notificationData: Omit<Notification, 'id' | 'createdAt' | 'read'>): Promise<Notification> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const newNotification: Notification = {
          ...notificationData,
          id: Date.now().toString(),
          createdAt: new Date().toISOString(),
          read: false,
        };
        
        notifications.push(newNotification);
        resolve(newNotification);
      }, 300);
    });
  },

  // Mark notification as read
  markAsRead: async (notificationId: string): Promise<boolean> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = notifications.findIndex(n => n.id === notificationId);
        if (index !== -1) {
          notifications[index].read = true;
          resolve(true);
        } else {
          resolve(false);
        }
      }, 300);
    });
  },

  // Mark all notifications as read for a user
  markAllAsRead: async (userId: string): Promise<number> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        let count = 0;
        notifications.forEach(notification => {
          if (notification.userId === userId && !notification.read) {
            notification.read = true;
            count++;
          }
        });
        resolve(count);
      }, 500);
    });
  },

  // Delete notification
  deleteNotification: async (notificationId: string): Promise<boolean> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const index = notifications.findIndex(n => n.id === notificationId);
        if (index !== -1) {
          notifications.splice(index, 1);
          resolve(true);
        } else {
          resolve(false);
        }
      }, 300);
    });
  },

  // Clear all notifications for a user
  clearAllNotifications: async (userId: string): Promise<number> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const initialCount = notifications.length;
        notifications = notifications.filter(n => n.userId !== userId);
        const deletedCount = initialCount - notifications.length;
        resolve(deletedCount);
      }, 500);
    });
  },

  // Get notification count
  getNotificationCount: async (userId: string): Promise<{
    total: number;
    unread: number;
    byCategory: {
      [key: string]: number;
    };
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const userNotifications = notifications.filter(n => n.userId === userId);
        const unreadNotifications = userNotifications.filter(n => !n.read);
        
        const byCategory: { [key: string]: number } = {};
        unreadNotifications.forEach(n => {
          if (!byCategory[n.category]) {
            byCategory[n.category] = 0;
          }
          byCategory[n.category]++;
        });
        
        resolve({
          total: userNotifications.length,
          unread: unreadNotifications.length,
          byCategory,
        });
      }, 300);
    });
  },

  // Send system notification to multiple users
  broadcastNotification: async (
    userIds: string[],
    notificationData: Omit<Notification, 'id' | 'userId' | 'createdAt' | 'read'>
  ): Promise<Notification[]> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        const createdNotifications: Notification[] = [];
        
        userIds.forEach(userId => {
          const notification: Notification = {
            ...notificationData,
            id: `${Date.now()}_${userId}`,
            userId,
            createdAt: new Date().toISOString(),
            read: false,
          };
          
          notifications.push(notification);
          createdNotifications.push(notification);
        });
        
        resolve(createdNotifications);
      }, 500);
    });
  },

  // Auto-generate notifications based on system events
  generateSystemNotifications: {
    // Leave request submitted
    leaveRequestSubmitted: async (employeeId: string, leaveId: string, employeeName: string): Promise<void> => {
      const hrUsers = ['1', '2']; // Admin and HR users
      const notificationData = {
        type: 'info' as const,
        category: 'leave' as const,
        title: 'New Leave Request',
        message: `${employeeName} has submitted a new leave request`,
        actionUrl: `/leaves?id=${leaveId}`,
        metadata: { leaveId, employeeId }
      };
      
      await notificationService.broadcastNotification(hrUsers, notificationData);
    },

    // Leave approved/rejected
    leaveStatusUpdated: async (employeeId: string, status: string, comments?: string): Promise<void> => {
      const notificationData = {
        type: status === 'approved' ? 'success' as const : 'error' as const,
        category: 'leave' as const,
        title: `Leave Request ${status}`,
        message: status === 'approved' 
          ? `Your leave request has been approved${comments ? `: ${comments}` : ''}`
          : `Your leave request has been rejected${comments ? `: ${comments}` : ''}`,
        actionUrl: '/leaves',
      };
      
      await notificationService.createNotification({
        ...notificationData,
        userId: employeeId,
      });
    },

    // Payroll processed
    payrollProcessed: async (employeeId: string, month: string, amount: number): Promise<void> => {
      const notificationData = {
        type: 'success' as const,
        category: 'payroll' as const,
        title: 'Payroll Processed',
        message: `Your ${month} payroll of $${amount.toLocaleString()} has been processed`,
        actionUrl: '/payroll',
        metadata: { month, amount }
      };
      
      await notificationService.createNotification({
        ...notificationData,
        userId: employeeId,
      });
    },

    // Attendance reminder
    attendanceReminder: async (employeeId: string): Promise<void> => {
      const notificationData = {
        type: 'warning' as const,
        category: 'attendance' as const,
        title: 'Attendance Reminder',
        message: 'Don\'t forget to clock in for today',
        actionUrl: '/attendance',
      };
      
      await notificationService.createNotification({
        ...notificationData,
        userId: employeeId,
      });
    },

    // Birthday reminder
    birthdayReminder: async (employeeId: string, birthdayEmployeeName: string): Promise<void> => {
      const notificationData = {
        type: 'info' as const,
        category: 'system' as const,
        title: 'Birthday Reminder',
        message: `Today is ${birthdayEmployeeName}'s birthday! 🎉`,
        metadata: { birthdayEmployee: birthdayEmployeeName }
      };
      
      // Send to all HR users
      const hrUsers = ['1', '2'];
      await notificationService.broadcastNotification(hrUsers, notificationData);
    },

    // System maintenance
    systemMaintenance: async (message: string, scheduledTime: string): Promise<void> => {
      const allUserIds = ['1', '2', '3', '4']; // In real app, get all user IDs
      const notificationData = {
        type: 'warning' as const,
        category: 'system' as const,
        title: 'Scheduled Maintenance',
        message: `${message} Scheduled for: ${scheduledTime}`,
        metadata: { scheduledTime }
      };
      
      await notificationService.broadcastNotification(allUserIds, notificationData);
    },
  },

  // Get notification preferences for user
  getNotificationPreferences: async (userId: string): Promise<{
    email: boolean;
    push: boolean;
    categories: {
      [key: string]: boolean;
    };
  }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        // Mock preferences - in real app, this would be stored per user
        resolve({
          email: true,
          push: true,
          categories: {
            leave: true,
            payroll: true,
            attendance: false,
            system: true,
            reminder: true,
          },
        });
      }, 300);
    });
  },

  // Update notification preferences
  updateNotificationPreferences: async (
    userId: string,
    preferences: {
      email?: boolean;
      push?: boolean;
      categories?: { [key: string]: boolean };
    }
  ): Promise<{ success: boolean }> => {
    return new Promise((resolve) => {
      setTimeout(() => {
        // Mock update - in real app, this would update user preferences
        resolve({ success: true });
      }, 300);
    });
  },
};