import React, { useState, useEffect } from 'react';
import {
  Box,
  Paper,
  Grid,
  Card,
  CardContent,
  Typography,
  Button,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogActions,
  TextField,
  IconButton,
  Fab,
  Tooltip,
  Avatar,
  Chip,
} from '@mui/material';
import {
  Add,
  Edit,
  Delete,
  People,
  AttachMoney,
  Business,
} from '@mui/icons-material';
import { Department } from '../types';
import { useFormValidation, VALIDATION_SCHEMAS } from '../utils/validation';

const mockDepartments: Department[] = [
  {
    id: '1',
    name: 'Engineering',
    description: 'Software development and technical operations',
    managerId: '1',
    employeeCount: 45,
    budget: 2500000,
  },
  {
    id: '2',
    name: 'Sales',
    description: 'Revenue generation and client acquisition',
    managerId: '2',
    employeeCount: 32,
    budget: 1800000,
  },
  {
    id: '3',
    name: 'Marketing',
    description: 'Brand promotion and market research',
    managerId: '3',
    employeeCount: 28,
    budget: 1200000,
  },
  {
    id: '4',
    name: 'Human Resources',
    description: 'Employee management and organizational development',
    managerId: '4',
    employeeCount: 15,
    budget: 800000,
  },
  {
    id: '5',
    name: 'Finance',
    description: 'Financial planning and accounting operations',
    managerId: '5',
    employeeCount: 25,
    budget: 1000000,
  },
];

interface DepartmentDialogProps {
  open: boolean;
  department: Department | null;
  onClose: () => void;
  onSave: (department: Partial<Department>) => void;
  mode: 'edit' | 'create';
}

const DepartmentDialog: React.FC<DepartmentDialogProps> = ({
  open,
  department,
  onClose,
  onSave,
  mode,
}) => {
  const [formData, setFormData] = useState<Partial<Department>>({
    name: '',
    description: '',
    managerId: '',
    budget: 0,
  });

  const {
    errors,
    validateForm,
    handleFieldChange,
    clearErrors,
    hasError,
    getErrorMessage,
  } = useFormValidation(VALIDATION_SCHEMAS.DEPARTMENT);

  useEffect(() => {
    if (department) {
      setFormData(department);
    } else if (mode === 'create') {
      setFormData({
        name: '',
        description: '',
        managerId: '',
        budget: 0,
      });
    }
  }, [department, mode]);

  const handleFieldChangeWithValidation = (fieldName: string, value: any) => {
    setFormData(prev => ({ ...prev, [fieldName]: value }));
    handleFieldChange(fieldName, value);
  };

  const handleSubmit = () => {
    const validation = validateForm(formData);
    if (!validation.isValid) {
      return;
    }
    onSave(formData);
    clearErrors();
  };

  return (
    <Dialog open={open} onClose={onClose} maxWidth="sm" fullWidth>
      <DialogTitle>
        {mode === 'create' ? 'Add New Department' : 'Edit Department'}
      </DialogTitle>
      <DialogContent>
        <Grid container spacing={2} sx={{ mt: 1 }}>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Department Name"
              value={formData.name || ''}
              onChange={(e) => handleFieldChangeWithValidation('name', e.target.value)}
              required
              error={hasError('name')}
              helperText={getErrorMessage('name')}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Description"
              multiline
              rows={3}
              value={formData.description || ''}
              onChange={(e) => handleFieldChangeWithValidation('description', e.target.value)}
              error={hasError('description')}
              helperText={getErrorMessage('description')}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Manager ID"
              value={formData.managerId || ''}
              onChange={(e) => handleFieldChangeWithValidation('managerId', e.target.value)}
              error={hasError('managerId')}
              helperText={hasError('managerId') ? getErrorMessage('managerId') : "Employee ID of the department manager"}
            />
          </Grid>
          <Grid size={12}>
            <TextField
              fullWidth
              label="Budget"
              type="number"
              value={formData.budget || ''}
              onChange={(e) => handleFieldChangeWithValidation('budget', Number(e.target.value))}
              required
              error={hasError('budget')}
              helperText={getErrorMessage('budget')}
              InputProps={{
                startAdornment: '$',
              }}
            />
          </Grid>
        </Grid>
      </DialogContent>
      <DialogActions>
        <Button onClick={onClose}>Cancel</Button>
        <Button onClick={handleSubmit} variant="contained">
          {mode === 'create' ? 'Add Department' : 'Save Changes'}
        </Button>
      </DialogActions>
    </Dialog>
  );
};

export const Departments: React.FC = () => {
  const [departments, setDepartments] = useState<Department[]>(mockDepartments);
  const [selectedDepartment, setSelectedDepartment] = useState<Department | null>(null);
  const [dialogOpen, setDialogOpen] = useState(false);
  const [dialogMode, setDialogMode] = useState<'edit' | 'create'>('create');

  const handleEdit = (department: Department) => {
    setSelectedDepartment(department);
    setDialogMode('edit');
    setDialogOpen(true);
  };

  const handleCreate = () => {
    setSelectedDepartment(null);
    setDialogMode('create');
    setDialogOpen(true);
  };

  const handleDelete = (department: Department) => {
    if (window.confirm(`Are you sure you want to delete ${department.name} department?`)) {
      setDepartments(prev => prev.filter(d => d.id !== department.id));
    }
  };

  const handleSave = (departmentData: Partial<Department>) => {
    if (dialogMode === 'create') {
      const newDepartment: Department = {
        ...departmentData,
        id: Date.now().toString(),
        employeeCount: 0,
      } as Department;
      setDepartments(prev => [...prev, newDepartment]);
    } else if (dialogMode === 'edit' && selectedDepartment) {
      setDepartments(prev =>
        prev.map(d => d.id === selectedDepartment.id ? { ...d, ...departmentData } : d)
      );
    }
    setDialogOpen(false);
  };

  const totalEmployees = departments.reduce((sum, dept) => sum + dept.employeeCount, 0);
  const totalBudget = departments.reduce((sum, dept) => sum + dept.budget, 0);

  return (
    <Box>
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
        <Typography variant="h4" gutterBottom>
          Department Management
        </Typography>
        <Tooltip title="Add Department">
          <Fab color="primary" aria-label="add" onClick={handleCreate}>
            <Add />
          </Fab>
        </Tooltip>
      </Box>
      {/* Summary Cards */}
      <Grid container spacing={3} sx={{ mb: 3 }}>
        <Grid
          size={{
            xs: 12,
            md: 4
          }}>
          <Card>
            <CardContent sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
              <Box>
                <Typography color="textSecondary" gutterBottom>
                  Total Departments
                </Typography>
                <Typography variant="h4">
                  {departments.length}
                </Typography>
              </Box>
              <Avatar sx={{ bgcolor: 'primary.main' }}>
                <Business />
              </Avatar>
            </CardContent>
          </Card>
        </Grid>
        <Grid
          size={{
            xs: 12,
            md: 4
          }}>
          <Card>
            <CardContent sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
              <Box>
                <Typography color="textSecondary" gutterBottom>
                  Total Employees
                </Typography>
                <Typography variant="h4">
                  {totalEmployees}
                </Typography>
              </Box>
              <Avatar sx={{ bgcolor: 'success.main' }}>
                <People />
              </Avatar>
            </CardContent>
          </Card>
        </Grid>
        <Grid
          size={{
            xs: 12,
            md: 4
          }}>
          <Card>
            <CardContent sx={{ display: 'flex', alignItems: 'center', justifyContent: 'space-between' }}>
              <Box>
                <Typography color="textSecondary" gutterBottom>
                  Total Budget
                </Typography>
                <Typography variant="h4">
                  ${(totalBudget / 1000000).toFixed(1)}M
                </Typography>
              </Box>
              <Avatar sx={{ bgcolor: 'warning.main' }}>
                <AttachMoney />
              </Avatar>
            </CardContent>
          </Card>
        </Grid>
      </Grid>
      {/* Department Cards */}
      <Grid container spacing={3}>
        {departments.map((department) => (
          <Grid
            key={department.id}
            size={{
              xs: 12,
              sm: 6,
              md: 4
            }}>
            <Card sx={{ height: '100%', position: 'relative' }}>
              <CardContent>
                <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'flex-start', mb: 2 }}>
                  <Typography variant="h6" component="h3">
                    {department.name}
                  </Typography>
                  <Box>
                    <IconButton size="small" onClick={() => handleEdit(department)}>
                      <Edit fontSize="small" />
                    </IconButton>
                    <IconButton size="small" color="error" onClick={() => handleDelete(department)}>
                      <Delete fontSize="small" />
                    </IconButton>
                  </Box>
                </Box>

                <Typography variant="body2" color="textSecondary" sx={{ mb: 2, minHeight: 40 }}>
                  {department.description}
                </Typography>

                <Box sx={{ mb: 2 }}>
                  <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 1 }}>
                    <Typography variant="body2" color="textSecondary">
                      Employees
                    </Typography>
                    <Chip
                      label={department.employeeCount}
                      size="small"
                      color="primary"
                      variant="outlined"
                    />
                  </Box>
                  <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 1 }}>
                    <Typography variant="body2" color="textSecondary">
                      Budget
                    </Typography>
                    <Typography variant="body2" fontWeight="bold">
                      ${(department.budget / 1000).toLocaleString()}K
                    </Typography>
                  </Box>
                  <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                    <Typography variant="body2" color="textSecondary">
                      Manager ID
                    </Typography>
                    <Typography variant="body2">
                      {department.managerId || 'Not assigned'}
                    </Typography>
                  </Box>
                </Box>

                <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                  <People fontSize="small" color="action" />
                  <Typography variant="body2" color="textSecondary">
                    {department.employeeCount > 0 
                      ? `${Math.round(department.budget / department.employeeCount / 1000)}K avg per employee`
                      : 'No employees assigned'
                    }
                  </Typography>
                </Box>
              </CardContent>
            </Card>
          </Grid>
        ))}
      </Grid>
      <DepartmentDialog
        open={dialogOpen}
        department={selectedDepartment}
        onClose={() => setDialogOpen(false)}
        onSave={handleSave}
        mode={dialogMode}
      />
    </Box>
  );
};