import React, { useState, useEffect } from 'react';
import { LoaderOverlay } from '../Common';
import {
  Box,
  Typography,
  Card,
  CardContent,
  Button,
  Table,
  TableBody,
  TableCell,
  TableContainer,
  TableHead,
  TableRow,
  TablePagination,
  Paper,
  Chip,
  Avatar,
  LinearProgress,
  TextField,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Tabs,
  Tab,
  CircularProgress,
} from '@mui/material';
import {
  AccountBalance,
  LocalHospital,
  MonetizationOn,
  Receipt,
  Business,
  Calculate,
  Download,
  Refresh,
} from '@mui/icons-material';
import { ComplianceStats, ComplianceRecord, ComplianceRule, ComplianceFilter } from '../../types';
import { complianceService } from '../../services';

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

function TabPanel(props: TabPanelProps) {
  const { children, value, index, ...other } = props;
  return (
    <div
      role="tabpanel"
      hidden={value !== index}
      id={`compliance-tabpanel-${index}`}
      aria-labelledby={`compliance-tab-${index}`}
      {...other}
    >
      {value === index && <Box sx={{ p: 3 }}>{children}</Box>}
    </div>
  );
}

export const ComplianceTracker: React.FC = () => {
  const [stats, setStats] = useState<ComplianceStats | null>(null);
  const [records, setRecords] = useState<ComplianceRecord[]>([]);
  const [rules, setRules] = useState<ComplianceRule[]>([]);
  const [filter, setFilter] = useState<ComplianceFilter>({});
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState('');
  const [page, setPage] = useState(0);
  const [rowsPerPage, setRowsPerPage] = useState(10);
  const [tabValue, setTabValue] = useState(0);
  const [operationLoading, setOperationLoading] = useState<string | null>(null);

  useEffect(() => {
    loadData();
  }, [filter]);

  const loadData = async () => {
    try {
      setLoading(true);
      setOperationLoading('Loading compliance data...');
      const [statsRes, recordsRes, rulesRes] = await Promise.all([
        complianceService.getComplianceStats(),
        complianceService.getComplianceRecords(filter),
        complianceService.getComplianceRules()
      ]);
      
      setStats(statsRes);
      setRecords(recordsRes.data);
      setRules(rulesRes);
      setError('');
    } catch (err: any) {
      setError(err.message || 'Failed to load compliance data');
    } finally {
      setLoading(false);
      setOperationLoading(null);
    }
  };

  const handleCalculateCompliance = async () => {
    try {
      setOperationLoading('Calculating compliance...');
      await complianceService.calculateCompliance({
        employeeIds: [],
        month: String(new Date().getMonth() + 1).padStart(2, '0'),
        year: new Date().getFullYear(),
        forceRecalculate: true
      });
      loadData();
    } catch (err: any) {
      setError(err.message || 'Failed to calculate compliance');
    } finally {
      setOperationLoading(null);
    }
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      minimumFractionDigits: 0,
      maximumFractionDigits: 0,
    }).format(amount);
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'compliant': return 'success';
      case 'pending': return 'warning';
      case 'overdue': return 'error';
      default: return 'default';
    }
  };

  const ComplianceOverview = () => (
    <Box sx={{ display: 'grid', gridTemplateColumns: { xs: '1fr', sm: '1fr 1fr', md: 'repeat(4, 1fr)' }, gap: 3 }}>
      {stats && (
        <>
          <Card>
            <CardContent>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                <Avatar sx={{ bgcolor: 'primary.main' }}>
                  <AccountBalance />
                </Avatar>
                <Box sx={{ ml: 2 }}>
                  <Typography variant="h6">{formatCurrency(stats.monthlyTotals.pfEmployee + stats.monthlyTotals.pfEmployer)}</Typography>
                  <Typography color="textSecondary" variant="body2">
                    Total PF
                  </Typography>
                </Box>
              </Box>
              <LinearProgress 
                variant="determinate" 
                value={75} 
                sx={{ height: 6, borderRadius: 3 }}
              />
              <Typography variant="caption" color="textSecondary" sx={{ mt: 1 }}>
                Employee: {formatCurrency(stats.monthlyTotals.pfEmployee)} | Employer: {formatCurrency(stats.monthlyTotals.pfEmployer)}
              </Typography>
            </CardContent>
          </Card>

          <Card>
            <CardContent>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                <Avatar sx={{ bgcolor: 'success.main' }}>
                  <LocalHospital />
                </Avatar>
                <Box sx={{ ml: 2 }}>
                  <Typography variant="h6">{formatCurrency(stats.monthlyTotals.esiEmployee + stats.monthlyTotals.esiEmployer)}</Typography>
                  <Typography color="textSecondary" variant="body2">
                    Total ESI
                  </Typography>
                </Box>
              </Box>
              <LinearProgress 
                variant="determinate" 
                value={60} 
                color="success"
                sx={{ height: 6, borderRadius: 3 }}
              />
              <Typography variant="caption" color="textSecondary" sx={{ mt: 1 }}>
                Employee: {formatCurrency(stats.monthlyTotals.esiEmployee)} | Employer: {formatCurrency(stats.monthlyTotals.esiEmployer)}
              </Typography>
            </CardContent>
          </Card>

          <Card>
            <CardContent>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                <Avatar sx={{ bgcolor: 'warning.main' }}>
                  <Receipt />
                </Avatar>
                <Box sx={{ ml: 2 }}>
                  <Typography variant="h6">{formatCurrency(stats.monthlyTotals.tds)}</Typography>
                  <Typography color="textSecondary" variant="body2">
                    Total TDS
                  </Typography>
                </Box>
              </Box>
              <LinearProgress 
                variant="determinate" 
                value={45} 
                color="warning"
                sx={{ height: 6, borderRadius: 3 }}
              />
            </CardContent>
          </Card>

          <Card>
            <CardContent>
              <Box sx={{ display: 'flex', alignItems: 'center', mb: 2 }}>
                <Avatar sx={{ bgcolor: 'info.main' }}>
                  <MonetizationOn />
                </Avatar>
                <Box sx={{ ml: 2 }}>
                  <Typography variant="h6">{formatCurrency(stats.monthlyTotals.gratuity)}</Typography>
                  <Typography color="textSecondary" variant="body2">
                    Gratuity Provision
                  </Typography>
                </Box>
              </Box>
              <LinearProgress 
                variant="determinate" 
                value={30} 
                color="info"
                sx={{ height: 6, borderRadius: 3 }}
              />
            </CardContent>
          </Card>
        </>
      )}
    </Box>
  );

  const ComplianceRecords = () => (
    <Box>
      <Box sx={{ display: 'flex', gap: 2, mb: 3, flexWrap: 'wrap' }}>
        <TextField
          placeholder="Search records..."
          value={filter.search || ''}
          onChange={(e) => setFilter({ ...filter, search: e.target.value })}
          size="small"
          sx={{ minWidth: 200 }}
        />
        
        <FormControl size="small" sx={{ minWidth: 120 }}>
          <InputLabel>Status</InputLabel>
          <Select
            value={filter.status || ''}
            onChange={(e) => setFilter({ ...filter, status: e.target.value || undefined })}
            label="Status"
          >
            <MenuItem value="">All</MenuItem>
            <MenuItem value="compliant">Compliant</MenuItem>
            <MenuItem value="pending">Pending</MenuItem>
            <MenuItem value="overdue">Overdue</MenuItem>
          </Select>
        </FormControl>
        
        <FormControl size="small" sx={{ minWidth: 120 }}>
          <InputLabel>Type</InputLabel>
          <Select
            value={filter.type || ''}
            onChange={(e) => setFilter({ ...filter, type: e.target.value as any || undefined })}
            label="Type"
          >
            <MenuItem value="">All Types</MenuItem>
            <MenuItem value="pf">PF</MenuItem>
            <MenuItem value="esi">ESI</MenuItem>
            <MenuItem value="tds">TDS</MenuItem>
            <MenuItem value="gratuity">Gratuity</MenuItem>
          </Select>
        </FormControl>

        <Button
          variant="contained"
          startIcon={<Calculate />}
          onClick={handleCalculateCompliance}
          disabled={loading}
        >
          Calculate
        </Button>

        <Button
          variant="outlined"
          startIcon={<Refresh />}
          onClick={loadData}
          disabled={loading}
        >
          Refresh
        </Button>
      </Box>

      <TableContainer component={Paper}>
        <Table>
          <TableHead>
            <TableRow>
              <TableCell>Employee</TableCell>
              <TableCell>Month/Year</TableCell>
              <TableCell>PF Employee</TableCell>
              <TableCell>PF Employer</TableCell>
              <TableCell>ESI Employee</TableCell>
              <TableCell>ESI Employer</TableCell>
              <TableCell>TDS</TableCell>
              <TableCell>Status</TableCell>
            </TableRow>
          </TableHead>
          <TableBody>
            {records.slice(page * rowsPerPage, page * rowsPerPage + rowsPerPage).map((record) => (
              <TableRow key={record.id}>
                <TableCell>{record.employeeName}</TableCell>
                <TableCell>{record.month}/{record.year}</TableCell>
                <TableCell>{formatCurrency(record.pfEmployee)}</TableCell>
                <TableCell>{formatCurrency(record.pfEmployer)}</TableCell>
                <TableCell>{formatCurrency(record.esiEmployee)}</TableCell>
                <TableCell>{formatCurrency(record.esiEmployer)}</TableCell>
                <TableCell>{formatCurrency(record.tdsAmount)}</TableCell>
                <TableCell>
                  <Chip 
                    label={record.status} 
                    size="small" 
                    color={getStatusColor(record.status) as any}
                  />
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
        <TablePagination
          component="div"
          count={records.length}
          page={page}
          onPageChange={(_, newPage) => setPage(newPage)}
          rowsPerPage={rowsPerPage}
          onRowsPerPageChange={(e) => setRowsPerPage(parseInt(e.target.value, 10))}
        />
      </TableContainer>
    </Box>
  );

  const ComplianceRules = () => (
    <Box sx={{ display: 'grid', gridTemplateColumns: { xs: '1fr', md: 'repeat(2, 1fr)', lg: 'repeat(3, 1fr)' }, gap: 3 }}>
      {rules.map((rule) => (
        <Card key={rule.id}>
          <CardContent>
            <Typography variant="h6" gutterBottom>
              {rule.name}
            </Typography>
            <Typography variant="body2" color="text.secondary" gutterBottom>
              {rule.description}
            </Typography>
            <Typography variant="body2" sx={{ mt: 2 }}>
              Type: {rule.type}
            </Typography>
            <Typography variant="body2">
              Configuration: {JSON.stringify(rule.configuration, null, 2)}
            </Typography>
          </CardContent>
        </Card>
      ))}
    </Box>
  );

  if (loading && records.length === 0) {
    return (
      <Box sx={{ display: 'flex', justifyContent: 'center', p: 4 }}>
        <CircularProgress />
      </Box>
    );
  }

  return (
    <Box>
      {error && (
        <Box sx={{ mb: 2, p: 2, bgcolor: 'error.light', color: 'error.contrastText', borderRadius: 1 }}>
          {error}
        </Box>
      )}

      <Box sx={{ borderBottom: 1, borderColor: 'divider', mb: 3 }}>
        <Tabs value={tabValue} onChange={(_, newValue) => setTabValue(newValue)}>
          <Tab icon={<AccountBalance />} label="Overview" />
          <Tab icon={<Business />} label="Records" />
          <Tab icon={<Receipt />} label="Rules" />
        </Tabs>
      </Box>

      <TabPanel value={tabValue} index={0}>
        <ComplianceOverview />
      </TabPanel>
      <TabPanel value={tabValue} index={1}>
        <ComplianceRecords />
      </TabPanel>
      <TabPanel value={tabValue} index={2}>
        <ComplianceRules />
      </TabPanel>

      <LoaderOverlay
        open={Boolean(operationLoading)}
        message={operationLoading || 'Loading...'}
      />
    </Box>
  );
};