import React, { useState, useEffect } from 'react';
import {
  Box,
  Typography,
  Button,
  Card,
  CardContent,
  IconButton,
  Tabs,
  Tab,
  Alert,
  Snackbar,
  Fab,
  Menu,
  MenuItem,
  ListItemIcon,
  ListItemText,
  Dialog,
  DialogTitle,
  DialogContent,
  DialogContentText,
  DialogActions,
  Chip,
  List,
  ListItem,
  ListItemAvatar,
  Avatar,
  ListItemSecondaryAction,
  Divider,
  LinearProgress,
  TextField,
  InputAdornment,
  FormControl,
  InputLabel,
  Select,
} from '@mui/material';
import {
  Add,
  People,
  Security,
  Edit,
  Delete,
  MoreVert,
  Search,
  PersonAdd,
  Email,
  Phone,
  Work,
  Refresh,
} from '@mui/icons-material';
import { UserProfile, UserStats, Role, RoleStats } from '../types';
import { userManagementService } from '../services';
import { UserDialog } from '../components/UserManagement/UserDialog';
import { RoleDialog } from '../components/UserManagement/RoleDialog';

interface TabPanelProps {
  children?: React.ReactNode;
  index: number;
  value: number;
}

function TabPanel(props: TabPanelProps) {
  const { children, value, index, ...other } = props;
  return (
    <div
      role="tabpanel"
      hidden={value !== index}
      id={`user-management-tabpanel-${index}`}
      {...other}
    >
      {value === index && <Box>{children}</Box>}
    </div>
  );
}

export const UserManagement: React.FC = () => {
  const [tabValue, setTabValue] = useState(0);
  const [users, setUsers] = useState<UserProfile[]>([]);
  const [roles, setRoles] = useState<Role[]>([]);
  const [userStats, setUserStats] = useState<UserStats | null>(null);
  const [roleStats, setRoleStats] = useState<RoleStats | null>(null);
  const [loading, setLoading] = useState(false);
  const [snackbar, setSnackbar] = useState({ 
    open: false, 
    message: '', 
    severity: 'success' as 'success' | 'error' | 'warning' | 'info' 
  });

  // Dialog states
  const [userDialog, setUserDialog] = useState(false);
  const [roleDialog, setRoleDialog] = useState(false);
  const [editingUser, setEditingUser] = useState<UserProfile | null>(null);
  const [editingRole, setEditingRole] = useState<Role | null>(null);
  const [deleteDialog, setDeleteDialog] = useState(false);
  const [itemToDelete, setItemToDelete] = useState<{ type: 'user' | 'role'; item: UserProfile | Role | null }>({ type: 'user', item: null });

  // Filter states
  const [searchTerm, setSearchTerm] = useState('');
  const [roleFilter, setRoleFilter] = useState('all');
  const [statusFilter, setStatusFilter] = useState('all');

  // Menu states
  const [actionMenu, setActionMenu] = useState<{
    anchor: HTMLElement | null;
    user: UserProfile | null;
    role: Role | null;
  }>({ anchor: null, user: null, role: null });

  useEffect(() => {
    loadData();
  }, []);

  const loadData = async () => {
    setLoading(true);
    try {
      const [usersResult, rolesResult, userStatsResult, roleStatsResult] = await Promise.all([
        userManagementService.getAllUsers(),
        userManagementService.getAllRoles(),
        userManagementService.getUserStats(),
        userManagementService.getRoleStats()
      ]);

      setUsers(usersResult.users);
      setRoles(rolesResult.roles);
      setUserStats(userStatsResult);
      setRoleStats(roleStatsResult);
    } catch (error) {
      showSnackbar('Failed to load data', 'error');
      console.error('Load data error:', error);
    } finally {
      setLoading(false);
    }
  };

  const showSnackbar = (message: string, severity: 'success' | 'error' | 'warning' | 'info') => {
    setSnackbar({ open: true, message, severity });
  };

  const handleAddUser = () => {
    setEditingUser(null);
    setUserDialog(true);
  };

  const handleAddRole = () => {
    setEditingRole(null);
    setRoleDialog(true);
  };

  const handleEditUser = (user: UserProfile) => {
    setEditingUser(user);
    setUserDialog(true);
    handleCloseActionMenu();
  };

  const handleEditRole = (role: Role) => {
    setEditingRole(role);
    setRoleDialog(true);
    handleCloseActionMenu();
  };

  const handleDeleteUser = (user: UserProfile) => {
    setItemToDelete({ type: 'user', item: user });
    setDeleteDialog(true);
    handleCloseActionMenu();
  };

  const handleDeleteRole = (role: Role) => {
    setItemToDelete({ type: 'role', item: role });
    setDeleteDialog(true);
    handleCloseActionMenu();
  };

  const confirmDelete = async () => {
    if (!itemToDelete.item) return;

    try {
      let success = false;
      
      if (itemToDelete.type === 'user') {
        success = await userManagementService.deleteUser(itemToDelete.item.id);
        if (success) {
          showSnackbar('User deleted successfully', 'success');
        }
      } else {
        success = await userManagementService.deleteRole(itemToDelete.item.id);
        if (success) {
          showSnackbar('Role deleted successfully', 'success');
        }
      }
      
      if (success) {
        loadData();
      } else {
        showSnackbar('Failed to delete item', 'error');
      }
    } catch (error: any) {
      showSnackbar(error.message || 'Failed to delete item', 'error');
    }

    setDeleteDialog(false);
    setItemToDelete({ type: 'user', item: null });
  };

  const handleSaveUser = (user: UserProfile) => {
    showSnackbar(
      editingUser ? 'User updated successfully' : 'User created successfully',
      'success'
    );
    loadData();
  };

  const handleSaveRole = (role: Role) => {
    showSnackbar(
      editingRole ? 'Role updated successfully' : 'Role created successfully',
      'success'
    );
    loadData();
  };

  const handleActionMenuOpen = (event: React.MouseEvent<HTMLElement>, user?: UserProfile, role?: Role) => {
    setActionMenu({ anchor: event.currentTarget, user: user || null, role: role || null });
  };

  const handleCloseActionMenu = () => {
    setActionMenu({ anchor: null, user: null, role: null });
  };

  const getRoleColor = (roleName: string): "default" | "primary" | "secondary" | "error" | "info" | "success" | "warning" => {
    switch (roleName.toLowerCase()) {
      case 'admin': return 'error';
      case 'hr': return 'primary';
      case 'employee': return 'info';
      default: return 'default';
    }
  };

  const getFilteredUsers = () => {
    return users.filter(user => {
      const matchesSearch = user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
                           user.email.toLowerCase().includes(searchTerm.toLowerCase());
      const matchesRole = roleFilter === 'all' || user.role === roleFilter;
      const matchesStatus = statusFilter === 'all' || 
                           (statusFilter === 'active' && user.isActive) ||
                           (statusFilter === 'inactive' && !user.isActive);
      
      return matchesSearch && matchesRole && matchesStatus;
    });
  };

  const renderStatsCards = () => (
    <Box sx={{ display: 'flex', gap: 2, mb: 3, flexWrap: 'wrap' }}>
      {userStats && (
        <>
          <Card sx={{ flex: '1 1 200px', minWidth: 200 }}>
            <CardContent>
              <Typography color="textSecondary" gutterBottom>
                Total Users
              </Typography>
              <Typography variant="h4" component="div">
                {userStats.total}
              </Typography>
            </CardContent>
          </Card>

          <Card sx={{ flex: '1 1 200px', minWidth: 200 }}>
            <CardContent>
              <Typography color="textSecondary" gutterBottom>
                Active Users
              </Typography>
              <Typography variant="h4" component="div">
                {userStats.active}
              </Typography>
            </CardContent>
          </Card>

          <Card sx={{ flex: '1 1 200px', minWidth: 200 }}>
            <CardContent>
              <Typography color="textSecondary" gutterBottom>
                Recent Logins
              </Typography>
              <Typography variant="h4" component="div">
                {userStats.recentLogins}
              </Typography>
            </CardContent>
          </Card>
        </>
      )}

      {roleStats && (
        <Card sx={{ flex: '1 1 200px', minWidth: 200 }}>
          <CardContent>
            <Typography color="textSecondary" gutterBottom>
              Total Roles
            </Typography>
            <Typography variant="h4" component="div">
              {roleStats.total}
            </Typography>
          </CardContent>
        </Card>
      )}
    </Box>
  );

  const renderUsersTab = () => (
    <Box>
      {/* Filters */}
      <Card sx={{ mb: 2 }}>
        <CardContent>
          <Box sx={{ display: 'flex', gap: 2, flexWrap: 'wrap', alignItems: 'center' }}>
            <TextField
              size="small"
              placeholder="Search users..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
              InputProps={{
                startAdornment: (
                  <InputAdornment position="start">
                    <Search />
                  </InputAdornment>
                ),
              }}
              sx={{ flex: '1 1 250px' }}
            />

            <FormControl size="small" sx={{ minWidth: 120 }}>
              <InputLabel>Role</InputLabel>
              <Select
                value={roleFilter}
                onChange={(e) => setRoleFilter(e.target.value)}
                label="Role"
              >
                <MenuItem value="all">All Roles</MenuItem>
                <MenuItem value="admin">Admin</MenuItem>
                <MenuItem value="hr">HR</MenuItem>
                <MenuItem value="employee">Employee</MenuItem>
              </Select>
            </FormControl>

            <FormControl size="small" sx={{ minWidth: 120 }}>
              <InputLabel>Status</InputLabel>
              <Select
                value={statusFilter}
                onChange={(e) => setStatusFilter(e.target.value)}
                label="Status"
              >
                <MenuItem value="all">All</MenuItem>
                <MenuItem value="active">Active</MenuItem>
                <MenuItem value="inactive">Inactive</MenuItem>
              </Select>
            </FormControl>

            <Button
              variant="outlined"
              startIcon={<Refresh />}
              onClick={loadData}
            >
              Refresh
            </Button>

            <Button
              variant="contained"
              startIcon={<PersonAdd />}
              onClick={handleAddUser}
            >
              Add User
            </Button>
          </Box>
        </CardContent>
      </Card>

      {/* Users List */}
      <Card>
        <CardContent>
          {loading && <LinearProgress sx={{ mb: 2 }} />}

          <List>
            {getFilteredUsers().map((user, index) => (
              <React.Fragment key={user.id}>
                <ListItem>
                  <ListItemAvatar>
                    <Avatar src={user.avatar}>
                      {user.name.charAt(0)}
                    </Avatar>
                  </ListItemAvatar>
                  
                  <ListItemText
                    primary={
                      <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 0.5 }}>
                        <Typography variant="subtitle1" fontWeight="medium">
                          {user.name}
                        </Typography>
                        <Chip 
                          label={user.role} 
                          size="small" 
                          color={getRoleColor(user.role)}
                        />
                        {!user.isActive && (
                          <Chip label="Inactive" size="small" color="default" />
                        )}
                      </Box>
                    }
                    secondary={
                      <Box>
                        <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 0.5 }}>
                          <Email fontSize="small" />
                          <Typography variant="body2">{user.email}</Typography>
                        </Box>
                        {user.phone && (
                          <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 0.5 }}>
                            <Phone fontSize="small" />
                            <Typography variant="body2">{user.phone}</Typography>
                          </Box>
                        )}
                        {user.department && (
                          <Box sx={{ display: 'flex', alignItems: 'center', gap: 1 }}>
                            <Work fontSize="small" />
                            <Typography variant="body2">
                              {user.position} • {user.department}
                            </Typography>
                          </Box>
                        )}
                      </Box>
                    }
                  />

                  <ListItemSecondaryAction>
                    <IconButton
                      onClick={(e) => handleActionMenuOpen(e, user)}
                      edge="end"
                    >
                      <MoreVert />
                    </IconButton>
                  </ListItemSecondaryAction>
                </ListItem>
                {index < getFilteredUsers().length - 1 && <Divider />}
              </React.Fragment>
            ))}
          </List>

          {getFilteredUsers().length === 0 && !loading && (
            <Box sx={{ textAlign: 'center', py: 4 }}>
              <People sx={{ fontSize: 64, color: 'text.secondary', mb: 2 }} />
              <Typography variant="h6" color="textSecondary" gutterBottom>
                No users found
              </Typography>
              <Typography variant="body2" color="textSecondary" sx={{ mb: 2 }}>
                {searchTerm || roleFilter !== 'all' || statusFilter !== 'all'
                  ? 'Try adjusting your filters'
                  : 'Add your first user to get started'
                }
              </Typography>
              {!searchTerm && roleFilter === 'all' && statusFilter === 'all' && (
                <Button
                  variant="contained"
                  startIcon={<PersonAdd />}
                  onClick={handleAddUser}
                >
                  Add User
                </Button>
              )}
            </Box>
          )}
        </CardContent>
      </Card>
    </Box>
  );

  const renderRolesTab = () => (
    <Box>
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
        <Typography variant="h6">Role Management</Typography>
        <Button
          variant="contained"
          startIcon={<Add />}
          onClick={handleAddRole}
        >
          Add Role
        </Button>
      </Box>

      <Card>
        <CardContent>
          {loading && <LinearProgress sx={{ mb: 2 }} />}

          <List>
            {roles.map((role, index) => (
              <React.Fragment key={role.id}>
                <ListItem>
                  <ListItemAvatar>
                    <Avatar sx={{ bgcolor: role.isSystem ? 'error.main' : 'primary.main' }}>
                      <Security />
                    </Avatar>
                  </ListItemAvatar>
                  
                  <ListItemText
                    primary={
                      <Box sx={{ display: 'flex', alignItems: 'center', gap: 1, mb: 0.5 }}>
                        <Typography variant="subtitle1" fontWeight="medium">
                          {role.name}
                        </Typography>
                        {role.isSystem && (
                          <Chip label="System" size="small" color="error" />
                        )}
                        {!role.isActive && (
                          <Chip label="Inactive" size="small" color="default" />
                        )}
                      </Box>
                    }
                    secondary={
                      <Box>
                        <Typography variant="body2" sx={{ mb: 1 }}>
                          {role.description}
                        </Typography>
                        <Typography variant="caption" color="text.secondary">
                          {role.permissions.length} permissions • {role.userCount} users
                        </Typography>
                      </Box>
                    }
                  />

                  <ListItemSecondaryAction>
                    <IconButton
                      onClick={(e) => handleActionMenuOpen(e, undefined, role)}
                      edge="end"
                    >
                      <MoreVert />
                    </IconButton>
                  </ListItemSecondaryAction>
                </ListItem>
                {index < roles.length - 1 && <Divider />}
              </React.Fragment>
            ))}
          </List>
        </CardContent>
      </Card>
    </Box>
  );

  return (
    <Box sx={{ p: 3 }}>
      <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
        <Typography variant="h4" gutterBottom>
          User & Role Management
        </Typography>
      </Box>

      {renderStatsCards()}

      <Tabs
        value={tabValue}
        onChange={(_, newValue) => setTabValue(newValue)}
        indicatorColor="primary"
        textColor="primary"
        sx={{ mb: 3 }}
      >
        <Tab icon={<People />} label="Users" />
        <Tab icon={<Security />} label="Roles" />
      </Tabs>

      <TabPanel value={tabValue} index={0}>
        {renderUsersTab()}
      </TabPanel>

      <TabPanel value={tabValue} index={1}>
        {renderRolesTab()}
      </TabPanel>

      {/* Floating Action Button for Mobile */}
      <Fab
        color="primary"
        aria-label="add"
        sx={{ 
          position: 'fixed', 
          bottom: 16, 
          right: 16,
          display: { xs: 'flex', sm: 'none' }
        }}
        onClick={tabValue === 0 ? handleAddUser : handleAddRole}
      >
        <Add />
      </Fab>

      {/* User Dialog */}
      <UserDialog
        open={userDialog}
        onClose={() => setUserDialog(false)}
        onSave={handleSaveUser}
        user={editingUser}
        roles={roles}
      />

      {/* Role Dialog */}
      <RoleDialog
        open={roleDialog}
        onClose={() => setRoleDialog(false)}
        onSave={handleSaveRole}
        role={editingRole}
      />

      {/* Action Menu */}
      <Menu
        anchorEl={actionMenu.anchor}
        open={Boolean(actionMenu.anchor)}
        onClose={handleCloseActionMenu}
      >
        {actionMenu.user && (
          [
            <MenuItem key="edit" onClick={() => actionMenu.user && handleEditUser(actionMenu.user)}>
              <ListItemIcon>
                <Edit fontSize="small" />
              </ListItemIcon>
              <ListItemText>Edit</ListItemText>
            </MenuItem>,
            <MenuItem key="delete" onClick={() => actionMenu.user && handleDeleteUser(actionMenu.user)}>
              <ListItemIcon>
                <Delete fontSize="small" />
              </ListItemIcon>
              <ListItemText>Delete</ListItemText>
            </MenuItem>
          ]
        )}
        {actionMenu.role && (
          [
            <MenuItem key="edit" onClick={() => actionMenu.role && handleEditRole(actionMenu.role)}>
              <ListItemIcon>
                <Edit fontSize="small" />
              </ListItemIcon>
              <ListItemText>Edit</ListItemText>
            </MenuItem>,
            !actionMenu.role.isSystem && (
              <MenuItem key="delete" onClick={() => actionMenu.role && handleDeleteRole(actionMenu.role)}>
                <ListItemIcon>
                  <Delete fontSize="small" />
                </ListItemIcon>
                <ListItemText>Delete</ListItemText>
              </MenuItem>
            )
          ]
        )}
      </Menu>

      {/* Delete Confirmation Dialog */}
      <Dialog
        open={deleteDialog}
        onClose={() => setDeleteDialog(false)}
      >
        <DialogTitle>
          Delete {itemToDelete.type === 'user' ? 'User' : 'Role'}
        </DialogTitle>
        <DialogContent>
          <DialogContentText>
            Are you sure you want to delete {itemToDelete.type === 'user' ? 'user' : 'role'} "
            {itemToDelete.item?.name}"? This action cannot be undone.
            {itemToDelete.type === 'role' && itemToDelete.item && (itemToDelete.item as Role).userCount && (itemToDelete.item as Role).userCount! > 0 && (
              <Box sx={{ mt: 2, p: 2, bgcolor: 'warning.light', borderRadius: 1 }}>
                <Typography variant="body2" color="warning.dark">
                  Warning: This role is assigned to {(itemToDelete.item as Role).userCount} users. 
                  They will lose these permissions.
                </Typography>
              </Box>
            )}
          </DialogContentText>
        </DialogContent>
        <DialogActions>
          <Button onClick={() => setDeleteDialog(false)}>Cancel</Button>
          <Button onClick={confirmDelete} color="error" variant="contained">
            Delete
          </Button>
        </DialogActions>
      </Dialog>

      {/* Snackbar */}
      <Snackbar
        open={snackbar.open}
        autoHideDuration={6000}
        onClose={() => setSnackbar({ ...snackbar, open: false })}
      >
        <Alert 
          severity={snackbar.severity} 
          onClose={() => setSnackbar({ ...snackbar, open: false })}
        >
          {snackbar.message}
        </Alert>
      </Snackbar>
    </Box>
  );
};