import React, { useState } from 'react';
import {
  Box,
  TextField,
  InputAdornment,
  IconButton,
  Drawer,
  Typography,
  Button,
  FormControl,
  InputLabel,
  Select,
  MenuItem,
  Chip,
  useTheme,
  useMediaQuery,
  Paper,
  Divider,
} from '@mui/material';
import {
  Search,
  FilterList,
  Close,
  Clear,
} from '@mui/icons-material';

interface Filter {
  key: string;
  label: string;
  type: 'select' | 'date' | 'text';
  options?: Array<{ value: string; label: string }>;
  value: string;
}

interface MobileFiltersProps {
  searchValue: string;
  onSearchChange: (value: string) => void;
  filters: Filter[];
  onFilterChange: (key: string, value: string) => void;
  onClearFilters: () => void;
  placeholder?: string;
}

export const MobileFilters: React.FC<MobileFiltersProps> = ({
  searchValue,
  onSearchChange,
  filters,
  onFilterChange,
  onClearFilters,
  placeholder = "Search...",
}) => {
  const [filtersOpen, setFiltersOpen] = useState(false);
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  const activeFiltersCount = filters.filter(f => f.value && f.value !== '').length;

  if (isMobile) {
    return (
      <>
        <Paper sx={{ p: 2, mb: 2 }}>
          <Box sx={{ display: 'flex', gap: 1, alignItems: 'center' }}>
            <TextField
              fullWidth
              size="small"
              placeholder={placeholder}
              value={searchValue}
              onChange={(e) => onSearchChange(e.target.value)}
              InputProps={{
                startAdornment: (
                  <InputAdornment position="start">
                    <Search fontSize="small" />
                  </InputAdornment>
                ),
                endAdornment: searchValue && (
                  <InputAdornment position="end">
                    <IconButton
                      size="small"
                      onClick={() => onSearchChange('')}
                    >
                      <Close fontSize="small" />
                    </IconButton>
                  </InputAdornment>
                ),
              }}
              sx={{
                '& .MuiOutlinedInput-root': {
                  fontSize: '16px', // Prevents zoom on iOS
                },
              }}
            />
            <IconButton
              color="primary"
              onClick={() => setFiltersOpen(true)}
              sx={{
                position: 'relative',
                minWidth: 48,
                height: 40,
              }}
            >
              <FilterList />
              {activeFiltersCount > 0 && (
                <Box
                  sx={{
                    position: 'absolute',
                    top: 4,
                    right: 4,
                    width: 16,
                    height: 16,
                    borderRadius: '50%',
                    bgcolor: 'error.main',
                    color: 'white',
                    fontSize: '0.7rem',
                    display: 'flex',
                    alignItems: 'center',
                    justifyContent: 'center',
                  }}
                >
                  {activeFiltersCount}
                </Box>
              )}
            </IconButton>
          </Box>

          {/* Active Filters Display */}
          {activeFiltersCount > 0 && (
            <Box sx={{ mt: 1, display: 'flex', flexWrap: 'wrap', gap: 0.5 }}>
              {filters
                .filter(f => f.value && f.value !== '')
                .map(filter => (
                  <Chip
                    key={filter.key}
                    label={`${filter.label}: ${filter.value}`}
                    size="small"
                    onDelete={() => onFilterChange(filter.key, '')}
                    sx={{ fontSize: '0.7rem', height: 24 }}
                  />
                ))}
              <Chip
                label="Clear All"
                size="small"
                variant="outlined"
                onClick={onClearFilters}
                sx={{ fontSize: '0.7rem', height: 24 }}
              />
            </Box>
          )}
        </Paper>

        {/* Mobile Filter Drawer */}
        <Drawer
          anchor="bottom"
          open={filtersOpen}
          onClose={() => setFiltersOpen(false)}
          PaperProps={{
            sx: {
              borderTopLeftRadius: 16,
              borderTopRightRadius: 16,
              maxHeight: '80vh',
            },
          }}
        >
          <Box sx={{ p: 3, pb: 4 }}>
            <Box sx={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center', mb: 3 }}>
              <Typography variant="h6" fontWeight={600}>
                Filters
              </Typography>
              <IconButton onClick={() => setFiltersOpen(false)}>
                <Close />
              </IconButton>
            </Box>

            <Box sx={{ display: 'flex', flexDirection: 'column', gap: 3 }}>
              {filters.map((filter) => (
                <FormControl key={filter.key} fullWidth size="small">
                  <InputLabel>{filter.label}</InputLabel>
                  {filter.type === 'select' ? (
                    <Select
                      value={filter.value}
                      onChange={(e) => onFilterChange(filter.key, e.target.value)}
                      label={filter.label}
                    >
                      <MenuItem value="">All {filter.label}</MenuItem>
                      {filter.options?.map(option => (
                        <MenuItem key={option.value} value={option.value}>
                          {option.label}
                        </MenuItem>
                      ))}
                    </Select>
                  ) : (
                    <TextField
                      type={filter.type}
                      value={filter.value}
                      onChange={(e) => onFilterChange(filter.key, e.target.value)}
                      label={filter.label}
                      size="small"
                      InputLabelProps={filter.type === 'date' ? { shrink: true } : undefined}
                    />
                  )}
                </FormControl>
              ))}
            </Box>

            <Divider sx={{ my: 3 }} />

            <Box sx={{ display: 'flex', gap: 2 }}>
              <Button
                fullWidth
                variant="outlined"
                startIcon={<Clear />}
                onClick={() => {
                  onClearFilters();
                  setFiltersOpen(false);
                }}
              >
                Clear All
              </Button>
              <Button
                fullWidth
                variant="contained"
                onClick={() => setFiltersOpen(false)}
              >
                Apply Filters
              </Button>
            </Box>
          </Box>
        </Drawer>
      </>
    );
  }

  // Desktop filters
  return (
    <Paper sx={{ mb: 2, p: 2 }}>
      <Box sx={{ display: 'flex', gap: 2, alignItems: 'center', flexWrap: 'wrap' }}>
        <TextField
          size="small"
          placeholder={placeholder}
          value={searchValue}
          onChange={(e) => onSearchChange(e.target.value)}
          InputProps={{
            startAdornment: (
              <InputAdornment position="start">
                <Search />
              </InputAdornment>
            ),
          }}
          sx={{ minWidth: 250 }}
        />
        
        {filters.map((filter) => (
          <FormControl key={filter.key} size="small" sx={{ minWidth: 120 }}>
            <InputLabel>{filter.label}</InputLabel>
            {filter.type === 'select' ? (
              <Select
                value={filter.value}
                onChange={(e) => onFilterChange(filter.key, e.target.value)}
                label={filter.label}
              >
                <MenuItem value="">All {filter.label}</MenuItem>
                {filter.options?.map(option => (
                  <MenuItem key={option.value} value={option.value}>
                    {option.label}
                  </MenuItem>
                ))}
              </Select>
            ) : (
              <TextField
                type={filter.type}
                value={filter.value}
                onChange={(e) => onFilterChange(filter.key, e.target.value)}
                size="small"
                InputLabelProps={filter.type === 'date' ? { shrink: true } : undefined}
              />
            )}
          </FormControl>
        ))}
        
        <Button
          startIcon={<FilterList />}
          onClick={onClearFilters}
        >
          Clear Filters
        </Button>
      </Box>
    </Paper>
  );
};