import React from 'react';
import {
  Box,
  Typography,
  useTheme,
  useMediaQuery,
  Card,
  CardContent,
} from '@mui/material';
import {
  ResponsiveContainer,
  BarChart,
  Bar,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  PieChart,
  Pie,
  Cell,
  LineChart,
  Line,
  AreaChart,
  Area,
} from 'recharts';

const COLORS = ['#0088FE', '#00C49F', '#FFBB28', '#FF8042', '#8884D8'];

interface MobileChartProps {
  title: string;
  height?: number;
  children: React.ReactNode;
  fullWidth?: boolean;
}

export const MobileChart: React.FC<MobileChartProps> = ({ 
  title, 
  height = 300, 
  children,
  fullWidth = false 
}) => {
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));
  
  const chartHeight = isMobile ? Math.min(height, 250) : height;
  
  return (
    <Card sx={{ 
      height: 'auto',
      mb: 2,
      ...(fullWidth && { gridColumn: { xs: '1 / -1' } })
    }}>
      <CardContent sx={{ p: { xs: 2, sm: 3 } }}>
        <Typography 
          variant="h6" 
          gutterBottom 
          sx={{ 
            fontSize: { xs: '1rem', sm: '1.125rem' },
            fontWeight: 600,
            mb: { xs: 1, sm: 2 }
          }}
        >
          {title}
        </Typography>
        <Box sx={{ width: '100%', height: chartHeight }}>
          {children}
        </Box>
      </CardContent>
    </Card>
  );
};

interface ResponsiveBarChartProps {
  data: any[];
  dataKey: string;
  xAxisKey: string;
  title: string;
  color?: string;
}

export const ResponsiveBarChart: React.FC<ResponsiveBarChartProps> = ({
  data,
  dataKey,
  xAxisKey,
  title,
  color = '#1976d2'
}) => {
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  return (
    <MobileChart title={title}>
      <ResponsiveContainer width="100%" height="100%">
        <BarChart data={data} margin={{ top: 5, right: 5, left: 5, bottom: 5 }}>
          <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
          <XAxis 
            dataKey={xAxisKey} 
            fontSize={isMobile ? 10 : 12}
            tick={{ fontSize: isMobile ? 10 : 12 }}
          />
          <YAxis 
            fontSize={isMobile ? 10 : 12}
            tick={{ fontSize: isMobile ? 10 : 12 }}
          />
          <Tooltip 
            contentStyle={{
              fontSize: isMobile ? '0.75rem' : '0.875rem',
              border: 'none',
              borderRadius: '8px',
              boxShadow: '0 4px 12px rgba(0,0,0,0.15)',
            }}
          />
          <Bar dataKey={dataKey} fill={color} radius={[4, 4, 0, 0]} />
        </BarChart>
      </ResponsiveContainer>
    </MobileChart>
  );
};

interface ResponsivePieChartProps {
  data: any[];
  dataKey: string;
  nameKey: string;
  title: string;
}

export const ResponsivePieChart: React.FC<ResponsivePieChartProps> = ({
  data,
  dataKey,
  nameKey,
  title
}) => {
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  return (
    <MobileChart title={title}>
      <ResponsiveContainer width="100%" height="100%">
        <PieChart>
          <Pie
            data={data}
            cx="50%"
            cy="50%"
            labelLine={false}
            label={isMobile ? false : ({ name, percent }) => 
              `${name} ${((percent || 0) * 100).toFixed(0)}%`
            }
            outerRadius={isMobile ? 60 : 80}
            fill="#8884d8"
            dataKey={dataKey}
          >
            {data.map((entry, index) => (
              <Cell key={`cell-${index}`} fill={COLORS[index % COLORS.length]} />
            ))}
          </Pie>
          <Tooltip 
            formatter={(value, name) => [value, name]}
            contentStyle={{
              fontSize: isMobile ? '0.75rem' : '0.875rem',
              border: 'none',
              borderRadius: '8px',
              boxShadow: '0 4px 12px rgba(0,0,0,0.15)',
            }}
          />
        </PieChart>
      </ResponsiveContainer>
    </MobileChart>
  );
};

interface ResponsiveLineChartProps {
  data: any[];
  dataKey: string;
  xAxisKey: string;
  title: string;
  color?: string;
}

export const ResponsiveLineChart: React.FC<ResponsiveLineChartProps> = ({
  data,
  dataKey,
  xAxisKey,
  title,
  color = '#1976d2'
}) => {
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  return (
    <MobileChart title={title}>
      <ResponsiveContainer width="100%" height="100%">
        <LineChart data={data} margin={{ top: 5, right: 5, left: 5, bottom: 5 }}>
          <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
          <XAxis 
            dataKey={xAxisKey} 
            fontSize={isMobile ? 10 : 12}
            tick={{ fontSize: isMobile ? 10 : 12 }}
          />
          <YAxis 
            fontSize={isMobile ? 10 : 12}
            tick={{ fontSize: isMobile ? 10 : 12 }}
          />
          <Tooltip 
            contentStyle={{
              fontSize: isMobile ? '0.75rem' : '0.875rem',
              border: 'none',
              borderRadius: '8px',
              boxShadow: '0 4px 12px rgba(0,0,0,0.15)',
            }}
          />
          <Line 
            type="monotone" 
            dataKey={dataKey} 
            stroke={color} 
            strokeWidth={isMobile ? 2 : 3}
            dot={{ r: isMobile ? 3 : 4 }}
          />
        </LineChart>
      </ResponsiveContainer>
    </MobileChart>
  );
};

interface ResponsiveAreaChartProps {
  data: any[];
  dataKey: string;
  xAxisKey: string;
  title: string;
  color?: string;
}

export const ResponsiveAreaChart: React.FC<ResponsiveAreaChartProps> = ({
  data,
  dataKey,
  xAxisKey,
  title,
  color = '#1976d2'
}) => {
  const theme = useTheme();
  const isMobile = useMediaQuery(theme.breakpoints.down('md'));

  return (
    <MobileChart title={title}>
      <ResponsiveContainer width="100%" height="100%">
        <AreaChart data={data} margin={{ top: 5, right: 5, left: 5, bottom: 5 }}>
          <CartesianGrid strokeDasharray="3 3" stroke="#f0f0f0" />
          <XAxis 
            dataKey={xAxisKey} 
            fontSize={isMobile ? 10 : 12}
            tick={{ fontSize: isMobile ? 10 : 12 }}
          />
          <YAxis 
            fontSize={isMobile ? 10 : 12}
            tick={{ fontSize: isMobile ? 10 : 12 }}
          />
          <Tooltip 
            contentStyle={{
              fontSize: isMobile ? '0.75rem' : '0.875rem',
              border: 'none',
              borderRadius: '8px',
              boxShadow: '0 4px 12px rgba(0,0,0,0.15)',
            }}
          />
          <Area 
            type="monotone" 
            dataKey={dataKey} 
            stroke={color} 
            fill={color} 
            fillOpacity={0.3}
          />
        </AreaChart>
      </ResponsiveContainer>
    </MobileChart>
  );
};